/*
<:copyright-gpl 
 Copyright 2012 Arcadyan Technology 
 All Rights Reserved. 
 
 This program is free software; you can distribute it and/or modify it 
 under the terms of the GNU General Public License (Version 2) as 
 published by the Free Software Foundation. 
 
 This program is distributed in the hope it will be useful, but WITHOUT 
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License 
 for more details. 
 
 You should have received a copy of the GNU General Public License along 
 with this program; if not, write to the Free Software Foundation, Inc., 
 59 Temple Place - Suite 330, Boston MA 02111-1307, USA. 
:>
*/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/spinlock.h>
#include <linux/platform_device.h>
#include <linux/gpio.h>

#include <bcm63268_cpu.h>
#include <bcm63268_gpio.h>
#include <bcm63268_io.h>
#include <bcm63268_regs.h>

DEFINE_SPINLOCK(bcm63268_gpio_lock);
static u32 gpio_out_low, gpio_out_high;

static void bcm63268_gpio_set(struct gpio_chip *chip,
			     unsigned gpio, int val)
{
	u32 reg;
	u32 mask;
	u32 *v;
	unsigned long flags;

	if (gpio >= bcm63268_gpio_count())
		BUG();

	if (gpio < 32) {
		reg = GPIO_DATA_LO_REG;
		mask = 1 << gpio;
		v = &gpio_out_low;
	} else {
		reg = GPIO_DATA_HI_REG;
		mask = 1 << (gpio - 32);
		v = &gpio_out_high;
	}

	spin_lock_irqsave(&bcm63268_gpio_lock, flags);
	if (val)
		*v |= mask;
	else
		*v &= ~mask;
	bcm_gpio_writel(*v, reg);
	spin_unlock_irqrestore(&bcm63268_gpio_lock, flags);
}

static int bcm63268_gpio_get(struct gpio_chip *chip, unsigned gpio)
{
	u32 reg;
	u32 mask;

	if (gpio >= bcm63268_gpio_count())
		BUG();

	if (gpio < 32) {
		reg = GPIO_DATA_LO_REG;
		mask = 1 << gpio;
	} else {
		reg = GPIO_DATA_HI_REG;
		mask = 1 << (gpio - 32);
	}

	return !!(bcm_gpio_readl(reg) & mask);
}

static int bcm63268_gpio_set_direction(struct gpio_chip *chip,
				      unsigned gpio, int dir)
{
	u32 reg;
	u32 mask;
	u32 tmp;
	unsigned long flags;

	if (gpio >= bcm63268_gpio_count())
		BUG();

	if (gpio < 32) {
		reg = GPIO_CTL_LO_REG;
		mask = 1 << gpio;
	} else {
		reg = GPIO_CTL_HI_REG;
		mask = 1 << (gpio - 32);
	}

	spin_lock_irqsave(&bcm63268_gpio_lock, flags);
	tmp = bcm_gpio_readl(reg);
	if (dir == GPIO_DIR_IN)
		tmp &= ~mask;
	else
		tmp |= mask;
	bcm_gpio_writel(tmp, reg);
	spin_unlock_irqrestore(&bcm63268_gpio_lock, flags);

	return 0;
}

int bcm63268_gpio_direction_input(struct gpio_chip *chip, unsigned gpio)
{
	return bcm63268_gpio_set_direction(chip, gpio, GPIO_DIR_IN);
}
EXPORT_SYMBOL(bcm63268_gpio_direction_input);

int bcm63268_gpio_direction_output(struct gpio_chip *chip,
					 unsigned gpio, int value)
{
	bcm63268_gpio_set(chip, gpio, value);
	return bcm63268_gpio_set_direction(chip, gpio, GPIO_DIR_OUT);
}
EXPORT_SYMBOL(bcm63268_gpio_direction_output);


static struct gpio_chip bcm63268_gpio_chip = {
	.label			= "bcm63268-gpio",
	.direction_input	= bcm63268_gpio_direction_input,
	.direction_output	= bcm63268_gpio_direction_output,
	.get			= bcm63268_gpio_get,
	.set			= bcm63268_gpio_set,
	.base			= 0,
};

int __init bcm63268_gpio_init(void)
{
	u32 reg;

	reg = bcm_gpio_readl(GPIO_CTL_HI_REG);
	gpio_out_high = bcm_gpio_readl(GPIO_DATA_HI_REG);
	gpio_out_high &= reg;

	reg = bcm_gpio_readl(GPIO_CTL_LO_REG);
	gpio_out_low = bcm_gpio_readl(GPIO_DATA_LO_REG);
	gpio_out_low &= reg;
	
	bcm63268_gpio_chip.ngpio = bcm63268_gpio_count();
	pr_info("registering %d GPIOs\n", bcm63268_gpio_chip.ngpio);

	return gpiochip_add(&bcm63268_gpio_chip);
}

