/*
<:copyright-gpl 
 Copyright 2012 Arcadyan Technology 
 All Rights Reserved. 
 
 This program is free software; you can distribute it and/or modify it 
 under the terms of the GNU General Public License (Version 2) as 
 published by the Free Software Foundation. 
 
 This program is distributed in the hope it will be useful, but WITHOUT 
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License 
 for more details. 
 
 You should have received a copy of the GNU General Public License along 
 with this program; if not, write to the Free Software Foundation, Inc., 
 59 Temple Place - Suite 330, Boston MA 02111-1307, USA. 
:>
*/

#include <linux/kernel.h>
#include <linux/err.h>
#include <linux/module.h>
#include <linux/spinlock.h>
#include <linux/interrupt.h>
#include <linux/clk.h>
#include <linux/if.h>
#include <asm/time.h>
#include <bcm63268_cpu.h>
#include <bcm63268_io.h>
#include <bcm63268_regs.h>
#include <bcm63268_irq.h>
#include <bcm63268_clk.h>

static DEFINE_MUTEX(bcm63268_dying_gasp_mutex);

typedef void (*bcm63268_dying_gasp_cb_t)(void *arg);
typedef struct bcm63268_dying_gasp_list_s {
	struct list_head list;
	char name[IFNAMSIZ];
	bcm63268_dying_gasp_cb_t dying_gasp_cb;
	void *context;
} bcm63268_dying_gasp_list_t;

static bcm63268_dying_gasp_list_t *dying_gasp_list_head = NULL;

void bcm63268_dying_gasp_register(char *name, void *cbfn, void *context)
{
	struct list_head *pos, *tmp;
    bcm63268_dying_gasp_list_t *new_entry;
	int found;

    if (name == NULL || cbfn == NULL) {
		pr_err("bcm63268-dg: name or callback handler is empty\n");
        return;
    }

    if (strlen(name) > (IFNAMSIZ - 1)) {
		pr_warning("bcm63268-dg: name is too long\n");
		name[IFNAMSIZ - 1] = '\0';
    }

    new_entry= (bcm63268_dying_gasp_list_t *)kmalloc(sizeof(bcm63268_dying_gasp_list_t), GFP_KERNEL);
    memset(new_entry, 0, sizeof(bcm63268_dying_gasp_list_t));
    INIT_LIST_HEAD(&new_entry->list);
    strncpy(new_entry->name, name, IFNAMSIZ - 1);
    new_entry->dying_gasp_cb = (bcm63268_dying_gasp_cb_t)cbfn;
    new_entry->context = context;

    mutex_lock(&bcm63268_dying_gasp_mutex);
    if (dying_gasp_list_head == NULL) {
		pr_err("bcm63268-dg: list head is null\n");
		kfree(new_entry);
    } else {
    	found = 0;
    	list_for_each_safe(pos, tmp, &dying_gasp_list_head->list) {
			bcm63268_dying_gasp_list_t *entry;
        	entry = list_entry(pos, bcm63268_dying_gasp_list_t, list);
			if (strcmp(entry->name, name) == 0) {	
				found = 1;
				break;
			}
	    }

		if (found) {
			printk("bcm63268-dg: %s already registerd!\n", name);
			kfree(new_entry);
		} else {
			list_add(&new_entry->list, &dying_gasp_list_head->list);
			printk("bcm63268-dg: register %s\n", name);
		}
    }
    mutex_unlock(&bcm63268_dying_gasp_mutex);
}
EXPORT_SYMBOL(bcm63268_dying_gasp_register);

void bcm63268_dying_gasp_deregister(char *name)
{
	struct list_head *pos, *tmp;
    bcm63268_dying_gasp_list_t *entry;

    if (name == NULL) {
		pr_err("bcm63268-dg: dying gasp name is empty\n");
        return;
    }

    mutex_lock(&bcm63268_dying_gasp_mutex);
    if (dying_gasp_list_head == NULL) {
        pr_err("bcm63268-dg: list head is null\n");
    } else {
    	list_for_each_safe(pos, tmp, &dying_gasp_list_head->list) {
        	entry = list_entry(pos, bcm63268_dying_gasp_list_t, list);
			if (strcmp(entry->name, name) == 0) {
				list_del(pos);
				kfree(entry);
				printk("bcm63268-dg: deregister %s\n", name);
			}
	    }
    }
    mutex_unlock(&bcm63268_dying_gasp_mutex);
}
EXPORT_SYMBOL(bcm63268_dying_gasp_deregister);

static irqreturn_t bcm63268_dying_gasp_interrupt(int irq, void * dev_id)
{
	char c;
    struct list_head *pos;
    bcm63268_dying_gasp_list_t *entry = NULL;
	bcm63268_dying_gasp_list_t *wan_dg = NULL;
	struct clk *periph_clk = NULL;
	u32 rate;

	c = 'D';
	bcm_uart0_writel(c, UART_FIFO_REG);
	c = '%';
	bcm_uart0_writel(c, UART_FIFO_REG);
	c = 'G';
	bcm_uart0_writel(c, UART_FIFO_REG);

    /* first to turn off everything other than dsl or gpon */
    list_for_each(pos, &dying_gasp_list_head->list) {
        entry = list_entry(pos, bcm63268_dying_gasp_list_t, list);
		if ((strncmp(entry->name, "dsl", 3) == 0)
			|| (strncmp(entry->name, "gpon", 3) == 0)) {
			wan_dg = entry;
		} else {
			entry->dying_gasp_cb(entry->context);
		}
    }

    /* turn-off dsl or gpon now */
    if(wan_dg)
        wan_dg->dying_gasp_cb(wan_dg->context);

    /* 
     * reset and shutdown system 
     * set watchdog to fire in 1 sec incase power is restored before reset occurs
	 */
	if (!periph_clk)
		periph_clk = clk_get(NULL, "periph");

	if (IS_ERR(periph_clk)) {
		rate = 50000000;
	} else {
		rate = clk_get_rate(periph_clk);
	}
	bcm_wdt_writel((1000000 * (rate /1000000 )), WDT_DEFVAL_REG);
	bcm_wdt_writel(WDT_START_1, WDT_CTL_REG);
	bcm_wdt_writel(WDT_START_2, WDT_CTL_REG);

    /* power is going down, keep looping */
    while (1);
	
    return IRQ_HANDLED;
}

static int __init bcm63268_dying_gasp_init(void)
{
	int ret, irq;
	bcm63268_dying_gasp_list_t *new_node;

	if (dying_gasp_list_head != NULL) {
		printk("unable to initialize Dying Gasp handler!\n");
		return -1;
	}

	new_node = (bcm63268_dying_gasp_list_t *)kmalloc(sizeof(bcm63268_dying_gasp_list_t), GFP_KERNEL);
	memset(new_node, 0, sizeof(bcm63268_dying_gasp_list_t));
	INIT_LIST_HEAD(&new_node->list);
	dying_gasp_list_head = new_node;

	irq = bcm63268_get_irq_number(IRQ_DYING_GASP);
	ret = request_irq(irq, bcm63268_dying_gasp_interrupt, 0, "bcm63268-dg", NULL);
	if (ret) {
		printk(KERN_ERR "bcm63268_dying_gasp: failed to register irq\n");
		return ret;
	}

	return 0;
}

arch_initcall(bcm63268_dying_gasp_init);

