/*
<:copyright-gpl 
 Copyright 2012 Arcadyan Technology 
 All Rights Reserved. 
 
 This program is free software; you can distribute it and/or modify it 
 under the terms of the GNU General Public License (Version 2) as 
 published by the Free Software Foundation. 
 
 This program is distributed in the hope it will be useful, but WITHOUT 
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License 
 for more details. 
 
 You should have received a copy of the GNU General Public License along 
 with this program; if not, write to the Free Software Foundation, Inc., 
 59 Temple Place - Suite 330, Boston MA 02111-1307, USA. 
:>
*/
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/cpu.h>
#include <asm/cpu-info.h>
#include <asm/bootinfo.h>
#include <bcm63268_cpu.h>
#include <bcm63268_regs.h>
#include <bcm63268_io.h>
#include <bcm63268_irq.h>
#include <asm/mips-boards/prom.h>

const unsigned long *bcm63268_regs_base;
EXPORT_SYMBOL(bcm63268_regs_base);

const int *bcm63268_irqs;
EXPORT_SYMBOL(bcm63268_irqs);

#ifdef CONFIG_SMP
extern struct plat_smp_ops bcm63268_smp_ops;
#endif /* CONFIG_SMP */

extern void prom_printf(const char * fmt, ...);

static u32 bcm63268_cpu_id;
static u16 bcm63268_cpu_rev;
static unsigned int bcm63268_cpu_freq;
static unsigned int bcm63268_ubus_freq;
static unsigned int bcm63268_memory_size;

/*
 * 63268 register sets and irqs
 */
static const unsigned long bcm963268_cpu_speed_table[0x20] = {
    0, 0, 400, 320, 0, 0, 0, 0, 0, 0, 333, 400, 0, 0, 320, 400,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};

static const unsigned long bcm963268_ubus_speed_table[0x20] = {
	0, 0, 133, 133, 0, 0, 0, 0, 0, 0, 167, 167, 0, 0, 200, 200,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};

static const unsigned long bcm963268_regs_base[] = {
	[RSET_PERF]		= BCM_63268_PERF_BASE,
	[RSET_TIMER]	= BCM_63268_TIMER_BASE,
	[RSET_WDT]		= BCM_63268_WDT_BASE,
	[RSET_UART0]	= BCM_63268_UART0_BASE,
	[RSET_UART1]	= BCM_63268_UART1_BASE,
	[RSET_GPIO]		= BCM_63268_GPIO_BASE,
	[RSET_HSSPI]	= BCM_63268_HSSPI_BASE,
	[RSET_MISC]		= BCM_63268_MISC_BASE,
	[RSET_USB_EHCI]	= BCM_63268_USB_EHCI_BASE,
	[RSET_USB_OHCI]	= BCM_63268_USB_OHCI_BASE,
	[RSET_USBH_CFG]	= BCM_63268_USBH_CFG_BASE,
	[RSET_PCIE]		= BCM_63268_PCIE_BASE,
	[RSET_WLAN]		= BCM_63268_WLAN_BASE,
	[RSET_WLAN_SHIM]	= BCM_63268_WLAN_SHIM_BASE,
	[RSET_NAND]		= BCM_63268_NAND_BASE,
	[RSET_NAND_CACHE]	= BCM_63268_NAND_CACHE_BASE,
	[RSET_NAND_INTR]	= BCM_63268_NAND_INTR_BASE,
};

static const int bcm963268_irqs[] = {
	[IRQ_TIMER]			= BCM_63268_TIMER_IRQ,
	[IRQ_RING_OSC]		= BCM_63268_RING_OSC_IRQ,
	[IRQ_UART0]			= BCM_63268_UART0_IRQ,
	[IRQ_UART1]			= BCM_63268_UART1_IRQ,
	[IRQ_HSSPI]			= BCM_63268_HSSPI_IRQ,
	[IRQ_WLAN_GPIO]		= BCM_63268_WLAN_GPIO_IRQ,
	[IRQ_WLAN]			= BCM_63268_WLAN_IRQ,
	[IRQ_IPSEC]			= BCM_63268_IPSEC_IRQ,
	[IRQ_USB_OHCI]		= BCM_63268_USBH_OHCI_IRQ,
	[IRQ_USB_EHCI]		= BCM_63268_USBH_EHCI_IRQ,
	[IRQ_NAND]			= BCM_63268_NAND_IRQ,
	[IRQ_PCM]			= BCM_63268_PCM_IRQ,
	[IRQ_EPHY]			= BCM_63268_EPHY_IRQ,
	[IRQ_GPHY]			= BCM_63268_GPHY_IRQ,
	[IRQ_DYING_GASP]	= BCM_63268_DYING_GASP_IRQ,
	[IRQ_EPHY_ENERGY]	= BCM_63268_EPHY_ENERGY_IRQ_BASE,
	[IRQ_GPHY_ENERGY]	= BCM_63268_GPHY_ENERGY_IRQ_BASE,
	[IRQ_IPSEC_DMA0]	= BCM_63268_IPSEC_DMA0_IRQ,
	[IRQ_IPSEC_DMA1]	= BCM_63268_IPSEC_DMA1_IRQ,
	[IRQ_XDSL]			= BCM_63268_XDSL_IRQ,
	[IRQ_FAP]			= BCM_63268_FAP_IRQ_BASE,
	[IRQ_PCIE_RC]		= BCM_63268_PCIE_RC_IRQ,
	[IRQ_PCIE_EP]		= BCM_63268_PCIE_EP_IRQ,
	[IRQ_ROBOSW]		= BCM_63268_ROBOSW_IRQ,
	[IRQ_ETHSW_RX_DMA]	= BCM_63268_ETHSW_RX_DMA_IRQ_BASE,
	[IRQ_ETHSW_TX_DMA]	= BCM_63268_ETHSW_TX_DMA_IRQ_BASE,
	[IRQ_GMAC]			= BCM_63268_GMAC_IRQ,
	[IRQ_GMAC_DMA]		= BCM_63268_GMAC_DMA_IRQ_BASE,
	[IRQ_PCM_DMA]		= BCM_63268_PCM_DMA_IRQ_BASE,
	[IRQ_DECT]			= BCM_63268_DECT_IRQ_BASE,
	[IRQ_EXTERNAL]		= BCM_63268_EXTERNAL_IRQ_BASE,
	[IRQ_SAR]			= BCM_63268_SAR_IRQ,
	[IRQ_SAR_DMA0]		= BCM_63268_SAR_DMA_IRQ_BASE0,
	[IRQ_SAR_DMA1]		= BCM_63268_SAR_DMA_IRQ_BASE1,
	[IRQ_SAR_DMA2]		= BCM_63268_SAR_DMA_IRQ_BASE2,
};

u16 __bcm63268_get_cpu_id(void)
{
	return bcm63268_cpu_id;
}

EXPORT_SYMBOL(__bcm63268_get_cpu_id);

u16 bcm63268_get_cpu_rev(void)
{
	return bcm63268_cpu_rev;
}

EXPORT_SYMBOL(bcm63268_get_cpu_rev);

unsigned int bcm63268_get_cpu_freq(void)
{
	return bcm63268_cpu_freq;
}
EXPORT_SYMBOL(bcm63268_get_cpu_freq);

unsigned int bcm63268_get_ubus_freq(void)
{
	return bcm63268_ubus_freq;
}
EXPORT_SYMBOL(bcm63268_get_ubus_freq);

unsigned int bcm63268_get_memory_size(void)
{
	return bcm63268_memory_size;
}
EXPORT_SYMBOL(bcm63268_get_memory_size);

static unsigned int bcm63268_detect_cpu_clock(void)
{
	unsigned int mips_pll_fvco;

	/*
	 * frequency depends on PLL configuration:
	 */
	if (BCMCPU_IS_63268() || BCMCPU_IS_63168()) {
		mips_pll_fvco = bcm_misc_readl(MISC_STRAPBUS_REG) & MISCSTRAPBUS_MIPS_PLL_FVCO_MASK;
		mips_pll_fvco >>= MISCSTRAPBUS_MIPS_PLL_FVCO_SHIFT;
		return bcm963268_cpu_speed_table[mips_pll_fvco] * 1000000;
	}

	return 0;
}

static unsigned int bcm63268_detect_ubus_clock(void)
{
	unsigned int mips_pll_fvco;

	/*
	 * frequency depends on PLL configuration:
	 */
	if (BCMCPU_IS_63268() || BCMCPU_IS_63168()) {
		mips_pll_fvco = bcm_misc_readl(MISC_STRAPBUS_REG) & MISCSTRAPBUS_MIPS_PLL_FVCO_MASK;
		mips_pll_fvco >>= MISCSTRAPBUS_MIPS_PLL_FVCO_SHIFT;
		return bcm963268_ubus_speed_table[mips_pll_fvco] * 1000000;
	}

	return 0;
}

/*
 * attempt to detect the amount of memory installed
 */
static unsigned int bcm63268_detect_memory_size(void)
{
	char *memsize_str;

	/* otherwise look in the environment */
	memsize_str = prom_getenv("memsize");
	if (!memsize_str) {
		pr_warning("memsize not set in boot prom, set to default (32Mb)\n");
		bcm63268_memory_size = 0x02000000;
	} else {
		bcm63268_memory_size = (simple_strtol(memsize_str, NULL, 0)) << 20;
	}

	return bcm63268_memory_size;
}

extern void prom_printf(const char * fmt, ...);
void __init bcm63268_cpu_init(void)
{
	volatile unsigned int tmp;
	unsigned int expected_cpu_id;
	struct cpuinfo_mips *c = &current_cpu_data;

	/* soc registers location depends on cpu type */
	expected_cpu_id = 0;

	switch (c->cputype) {
	case CPU_BCM63268:
		expected_cpu_id = BCM63268_CPU_ID;
		bcm63268_regs_base = bcm963268_regs_base;
		bcm63268_irqs = bcm963268_irqs;
		break;
		
	case CPU_BCM63168:
		expected_cpu_id = BCM63168_CPU_ID;
		bcm63268_regs_base = bcm963268_regs_base;
		bcm63268_irqs = bcm963268_irqs;
		break;
		
	case CPU_BCM63167:
		expected_cpu_id = BCM63167_CPU_ID;
		bcm63268_regs_base = bcm963268_regs_base;
		bcm63268_irqs = bcm963268_irqs;
		break;
	}

	/*
	 * really early to panic, but delaying panic would not help since we
	 * will never get any working console
	 */
	if (!expected_cpu_id)
		panic("unsupported Broadcom CPU");

#ifdef CONFIG_SMP
	register_smp_ops(&bcm63268_smp_ops);
#endif /* CONFIG_SMP */
	
	/*
	 * bcm63268_regs_base is set, we can access soc registers
	 */

	/* double check CPU type */
	tmp = bcm_perf_readl(PERF_REV_REG);
	bcm63268_cpu_id = (tmp & REV_CHIPID_MASK) >> REV_CHIPID_SHIFT;
	bcm63268_cpu_rev = (tmp & REV_REVID_MASK) >> REV_REVID_SHIFT;

	if (bcm63268_cpu_id != expected_cpu_id) {
		panic("BCM63268 CPU id mismatch");
	}

	bcm63268_cpu_freq = bcm63268_detect_cpu_clock();
	bcm63268_ubus_freq = bcm63268_detect_ubus_clock();
	bcm63268_memory_size = bcm63268_detect_memory_size();

	printk(KERN_INFO "Detected Broadcom 0x%04x CPU revision %02x\n",
	       bcm63268_cpu_id, bcm63268_cpu_rev);
	printk(KERN_INFO "CPU frequency is %u MHz\n",
	       bcm63268_cpu_freq / 1000000);
	printk(KERN_INFO "%uMB of RAM installed\n",
	       bcm63268_memory_size >> 20);
}

