/*
<:copyright-gpl 
 Copyright 2012 Arcadyan Technology 
 All Rights Reserved. 
 
 This program is free software; you can distribute it and/or modify it 
 under the terms of the GNU General Public License (Version 2) as 
 published by the Free Software Foundation. 
 
 This program is distributed in the hope it will be useful, but WITHOUT 
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License 
 for more details. 
 
 You should have received a copy of the GNU General Public License along 
 with this program; if not, write to the Free Software Foundation, Inc., 
 59 Temple Place - Suite 330, Boston MA 02111-1307, USA. 
:>
*/

#include <linux/kernel.h>
#include <linux/err.h>
#include <linux/module.h>
#include <linux/spinlock.h>
#include <linux/interrupt.h>
#include <linux/clk.h>
#include <linux/cpufreq.h>
#include <linux/delay.h>
#include <linux/sched.h>
#include <linux/platform_device.h>
#include <asm/time.h>
#include <bcm63268_cpu.h>
#include <bcm63268_io.h>
#include <bcm63268_regs.h>
#include <bcm63268_irq.h>
#include <bcm63268_clk.h>
#include <bcm63268_pwrmngt.h>

static struct clk *cpuclk;

/* Minimum CLK support */
enum {
	DC_DISABLE = RATIO_ONE_SYNC,
	DC_13PT = RATIO_ONE_EIGHTH,
	DC_25PT = RATIO_ONE_QUARTER,
	DC_50PT = RATIO_ONE_HALF,
	DC_RESV
};

struct cpufreq_frequency_table bcm63268_cpufreq_freq_table[] = {
	{DC_DISABLE, 0},
	{DC_13PT, 0},
	{DC_25PT, 0},
	{DC_50PT, 0},
	{DC_RESV, CPUFREQ_TABLE_END},
};

static int bcm63268_cpu_freq_notifier(struct notifier_block *nb,
					unsigned long val, void *data)
{
	if (val == CPUFREQ_POSTCHANGE)
		current_cpu_data.udelay_val = loops_per_jiffy;

	return 0;
}

static unsigned int bcm63268_cpufreq_get(unsigned int cpu)
{
	return clk_get_rate(cpuclk);
}

/*
 * Here we notify other drivers of the proposed change and the final change.
 */
static int bcm63268_cpufreq_target(struct cpufreq_policy *policy,
				     unsigned int target_freq,
				     unsigned int relation)
{
	unsigned int cpu = policy->cpu;
	unsigned int newstate = 0;
	cpumask_t cpus_allowed;
	struct cpufreq_freqs freqs;
	unsigned int freq;

	if (!cpu_online(cpu))
		return -ENODEV;

	cpus_allowed = current->cpus_allowed;
	set_cpus_allowed_ptr(current, cpumask_of(cpu));

	if (cpufreq_frequency_table_target
	    (policy, &bcm63268_cpufreq_freq_table[0], target_freq, relation,
	     &newstate))
		return -EINVAL;

	freq = bcm63268_cpufreq_freq_table[newstate].frequency;
	if (freq < policy->min || freq > policy->max)
		return -EINVAL;

	pr_info("cpufreq: requested frequency %u Hz\n", target_freq * 1000);

	freqs.cpu = cpu;
	freqs.old = bcm63268_cpufreq_get(cpu);
	freqs.new = freq;
	freqs.flags = 0;

	if (freqs.new == freqs.old)
		return 0;

	/* notifiers */
	cpufreq_notify_transition(&freqs, CPUFREQ_PRECHANGE);

	set_cpus_allowed_ptr(current, &cpus_allowed);

	/* setting the cpu frequency */
	/* switch back to RATIO_ONE_ASYNC first */
    BcmPwrMngtSetASCR(RATIO_ONE_ASYNC);
	/* set the new requested ratio */
	if (bcm63268_cpufreq_freq_table[newstate].index == DC_DISABLE)
		BcmPwrMngtSetSCR();
	else
		BcmPwrMngtSetASCR(bcm63268_cpufreq_freq_table[newstate].index);
	/* update software status */
	clk_set_rate(cpuclk, freq);

	/* notifiers */
	cpufreq_notify_transition(&freqs, CPUFREQ_POSTCHANGE);

	pr_info("cpufreq: set frequency %u kHz\n", freq);

	return 0;
}

static int bcm63268_cpufreq_cpu_init(struct cpufreq_policy *policy)
{
	if (!cpu_online(policy->cpu))
		return -ENODEV;

	cpuclk = clk_get(NULL, "cpu");
	if (IS_ERR(cpuclk)) {
		printk(KERN_ERR "cpufreq: couldn't get CPU clk\n");
		return PTR_ERR(cpuclk);
	}

	cpuclk->rate = bcm63268_get_cpu_freq() / 1000;
	if (!cpuclk->rate)
		return -EINVAL;

	bcm63268_cpufreq_freq_table[0].frequency = cpuclk->rate;

	bcm63268_cpufreq_freq_table[1].frequency = cpuclk->rate / 8;

	bcm63268_cpufreq_freq_table[2].frequency = cpuclk->rate / 4;

	bcm63268_cpufreq_freq_table[3].frequency = cpuclk->rate / 2;
	
	policy->cur = cpuclk->rate;

	cpufreq_frequency_table_get_attr(&bcm63268_cpufreq_freq_table[0],
					 policy->cpu);

	return cpufreq_frequency_table_cpuinfo(policy,
					    &bcm63268_cpufreq_freq_table[0]);
}

static int bcm63268_cpufreq_verify(struct cpufreq_policy *policy)
{
	return cpufreq_frequency_table_verify(policy,
					      &bcm63268_cpufreq_freq_table[0]);
}

static int bcm63268_cpufreq_exit(struct cpufreq_policy *policy)
{
	clk_put(cpuclk);
	return 0;
}

static struct notifier_block bcm63268_cpufreq_notifier_block = {
	.notifier_call = bcm63268_cpu_freq_notifier
};

static struct freq_attr *bcm63268_cpufreq_attr_table[] = {
	&cpufreq_freq_attr_scaling_available_freqs,
	NULL,
};

static struct cpufreq_driver bcm63268_cpufreq_driver = {
	.owner = THIS_MODULE,
	.name = "bcm63268",
	.init = bcm63268_cpufreq_cpu_init,
	.verify = bcm63268_cpufreq_verify,
	.target = bcm63268_cpufreq_target,
	.get = bcm63268_cpufreq_get,
	.exit = bcm63268_cpufreq_exit,
	.attr = bcm63268_cpufreq_attr_table,
};

static struct platform_device_id bcm63268_cpufreq_platform_devid[] = {
	{
		.name = "bcm63268-cpufreq",
	},
	{}
};

MODULE_DEVICE_TABLE(platform, bcm63268_cpufreq_platform_devid);

static struct platform_driver bcm63268_cpufreq_platform_driver = {
	.driver = {
		.name = "bcm63268-cpufreq",
		.owner = THIS_MODULE,
	},
	.id_table = bcm63268_cpufreq_platform_devid,
};

static struct platform_device bcm63268_cpufreq_platform_device = {
	.name = "bcm63268-cpufreq",
	.id = -1,
};

static int __init bcm63268_cpu_freq_init(void)
{
	int ret;

	ret = platform_device_register(&bcm63268_cpufreq_platform_device);
	if (ret) {
		printk(KERN_ERR "bcm63268_cpu_freq: failed to register platform device\n");
		return ret;
	}

	ret = platform_driver_register(&bcm63268_cpufreq_platform_driver);
	if (ret) {
		printk(KERN_ERR "bcm63268_cpu_freq: failed to register platform driver\n");
		return ret;
	}

	cpufreq_register_notifier(&bcm63268_cpufreq_notifier_block, CPUFREQ_TRANSITION_NOTIFIER);

	ret = cpufreq_register_driver(&bcm63268_cpufreq_driver);
	if (ret) {
		printk(KERN_ERR "bcm63268_cpu_freq: failed to register cpufreq driver\n");
		return ret;
	}

	pr_info("bcm63268_cpu_freq: CPU frequency driver\n");

	return 0;
}

static void __exit bcm63268_cpu_freq_exit(void)
{
	cpufreq_unregister_driver(&bcm63268_cpufreq_driver);
	cpufreq_unregister_notifier(&bcm63268_cpufreq_notifier_block,
				    CPUFREQ_TRANSITION_NOTIFIER);

	platform_driver_unregister(&bcm63268_cpufreq_platform_driver);
}

module_init(bcm63268_cpu_freq_init);
module_exit(bcm63268_cpu_freq_exit);
MODULE_AUTHOR("Knight Chang <knight_chang@arcadyan.com.tw>");
MODULE_DESCRIPTION("cpufreq driver for BCM63268 SoCs");
MODULE_LICENSE("GPL");

