/*
<:copyright-gpl 
 Copyright 2012 Arcadyan Technology 
 All Rights Reserved. 
 
 This program is free software; you can distribute it and/or modify it 
 under the terms of the GNU General Public License (Version 2) as 
 published by the Free Software Foundation. 
 
 This program is distributed in the hope it will be useful, but WITHOUT 
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License 
 for more details. 
 
 You should have received a copy of the GNU General Public License along 
 with this program; if not, write to the Free Software Foundation, Inc., 
 59 Temple Place - Suite 330, Boston MA 02111-1307, USA. 
:>
*/

#include <linux/module.h>
#include <linux/mutex.h>
#include <linux/err.h>
#include <linux/clk.h>
#include <asm/delay.h>
#include <bcm63268_cpu.h>
#include <bcm63268_io.h>
#include <bcm63268_regs.h>
#include <bcm63268_clk.h>

static DEFINE_MUTEX(clocks_mutex);


static void clk_enable_unlocked(struct clk *clk)
{
	if (clk->set && (clk->usage++) == 0)
		clk->set(clk, 1);
}

static void clk_disable_unlocked(struct clk *clk)
{
	if (clk->set && (--clk->usage) == 0)
		clk->set(clk, 0);
}

static void bcm_hwclock_set(u32 mask, int enable)
{
	u32 reg;

	reg = bcm_perf_readl(PERF_CKCTL_REG);
	if (enable)
		reg |= mask;
	else
		reg &= ~mask;
	bcm_perf_writel(reg, PERF_CKCTL_REG);
}

/*
 * SPI clock
 */
static void spi_set(struct clk *clk, int enable)
{
	u32 mask;

	if (BCMCPU_IS_63268() || BCMCPU_IS_63168() || BCMCPU_IS_63167())
		mask = CKCTL_63268_HSSPI_EN;

	bcm_hwclock_set(mask, enable);
}

static struct clk clk_spi = {
	.set	= spi_set,
};

/*
 * WLAN-On-Chip clock
 */
static void woc_set(struct clk *clk, int enable)
{
	u32 mask;

	if (BCMCPU_IS_63268() || BCMCPU_IS_63168() || BCMCPU_IS_63167())
		mask = CKCTL_63268_WOC_EN;

	bcm_hwclock_set(mask, enable);
}

static struct clk clk_woc = {
	.set	= woc_set,
};

/*
 * PCI-E clock
 */
static void pcie_set(struct clk *clk, int enable)
{
	u32 mask;

	if (BCMCPU_IS_63268() || BCMCPU_IS_63168() || BCMCPU_IS_63167())
		mask = CKCTL_63268_PCIE_EN;

	bcm_hwclock_set(mask, enable);
}

static struct clk clk_pcie = {
	.set	= pcie_set,
};

/*
 * USB host controller clock
 */
static void usbh_set(struct clk *clk, int enable)
{
	u32 mask;

	if (BCMCPU_IS_63268() || BCMCPU_IS_63168() || BCMCPU_IS_63167())
		mask = CKCTL_63268_USBH_EN;

	bcm_hwclock_set(mask, enable);
}

static struct clk clk_usbh = {
	.set	= usbh_set,
};

/*
 * USB device controller clock
 */
static void usbd_set(struct clk *clk, int enable)
{
	u32 mask;

	if (BCMCPU_IS_63268() || BCMCPU_IS_63168() || BCMCPU_IS_63167())
		mask = CKCTL_63268_USBD_EN;

	bcm_hwclock_set(mask, enable);
}

static struct clk clk_usbd = {
	.set	= usbd_set,
};

/*
 * NAND FLASH controller clock
 */
static void nfc_set(struct clk *clk, int enable)
{
	u32 mask;

	if (BCMCPU_IS_63268() || BCMCPU_IS_63168() || BCMCPU_IS_63167())
		mask = CKCTL_63268_NAND_EN;

	bcm_hwclock_set(mask, enable);
}

static struct clk clk_nfc = {
	.set	= nfc_set,
};

/*
 * Internal peripheral clock
 */
static struct clk clk_periph = {
	.rate	= (50 * 1000 * 1000),
};

#if defined(CONFIG_BCM_ADSL) || defined(CONFIG_BCM_ADSL_MODULE)
/*
 * XDSL controller clock
 */
static void xdsl_set(struct clk *clk, int enable)
{
	u32 mask;

	if (BCMCPU_IS_63268() || BCMCPU_IS_63168() || BCMCPU_IS_63167())
		mask = CKCTL_63268_SAR_EN | CKCTL_63268_PHYMIPS_EN | CKCTL_63268_VDSL_EN | 
									CKCTL_63268_VDSL_AFE_EN | CKCTL_63268_VDSL_QPROC_EN;

	bcm_hwclock_set(mask, enable);
}
#endif /* CONFIG_BCM_ADSL || CONFIG_BCM_ADSL_MODULE */

static struct clk clk_xdsl = {
	.set	= xdsl_set,
};

#ifdef CONFIG_CPU_FREQ
/*
 * CPU clock
 */
static struct clk clk_cpu = {
	.rate	= 0, /* update in the bcm632x_cpufreq_cpu_init() */
};
#endif /* CONFIG_CPU_FREQ */

/*
 * Linux clock API implementation
 */
int clk_enable(struct clk *clk)
{
	mutex_lock(&clocks_mutex);
	clk_enable_unlocked(clk);
	mutex_unlock(&clocks_mutex);
	return 0;
}
EXPORT_SYMBOL(clk_enable);

void clk_disable(struct clk *clk)
{
	mutex_lock(&clocks_mutex);
	clk_disable_unlocked(clk);
	mutex_unlock(&clocks_mutex);
}
EXPORT_SYMBOL(clk_disable);

unsigned long clk_get_rate(struct clk *clk)
{
	return clk->rate;
}
EXPORT_SYMBOL(clk_get_rate);

int clk_set_rate(struct clk *clk, unsigned long rate)
{
	clk->rate = rate;

	return 0;
}
EXPORT_SYMBOL(clk_set_rate);

struct clk *clk_get(struct device *dev, const char *id)
{
	if (!strcmp(id, "spi"))
		return &clk_spi;
	if (!strcmp(id, "periph"))
		return &clk_periph;
	if (!strcmp(id, "woc"))
		return &clk_woc;
	if (!strcmp(id, "pcie"))
		return &clk_pcie;
	if (!strcmp(id, "usbd"))
		return &clk_usbd;
	if (!strcmp(id, "usbh"))
		return &clk_usbh;
	if (!strcmp(id, "nfc"))
		return &clk_nfc;
#if defined(CONFIG_BCM_ADSL) || defined(CONFIG_BCM_ADSL_MODULE)
	if (!strcmp(id, "xdsl"))
		return &clk_xdsl;
#endif /* CONFIG_BCM_ADSL || CONFIG_BCM_ADSL_MODULE */
#ifdef CONFIG_CPU_FREQ
	if (!strcmp(id, "cpu"))
		return &clk_cpu;
#endif /* CONFIG_CPU_FREQ */
	return ERR_PTR(-ENOENT);
}
EXPORT_SYMBOL(clk_get);

void clk_put(struct clk *clk)
{
	/* do nothing */
}
EXPORT_SYMBOL(clk_put);

