/*
<:copyright-gpl 
 Copyright 2013 Arcadyan Technology 
 All Rights Reserved. 
 
 This program is free software; you can distribute it and/or modify it 
 under the terms of the GNU General Public License (Version 2) as 
 published by the Free Software Foundation. 
 
 This program is distributed in the hope it will be useful, but WITHOUT 
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License 
 for more details. 
 
 You should have received a copy of the GNU General Public License along 
 with this program; if not, write to the Free Software Foundation, Inc., 
 59 Temple Place - Suite 330, Boston MA 02111-1307, USA. 
:>
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <errno.h>
#include <net/if.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <linux/netfilter.h>
#include "libnetfilter_queue/libnetfilter_queue.h"
#include "nfqueue_logger.h"

static char nfqueue_logger_target[256];
static int nfqueue_logger_exit = 0;
static nfqueue_logger_cb_data_t nfqueue_logger_cb_data;

static void nfqueue_logger_netlink_loop(void)
{
	struct nfq_handle *h;
	struct nfq_q_handle *qh;
	struct nfnl_handle *nh;
	int fd, rv;
	char *buf;
	int buf_size;
	nfqueue_logger_cb_t nfqueue_logger_cb;
	struct timeval wait_time;
	fd_set rfds;
	int n, bindAgain = 0;

	switch (nfqueue_logger_cb_data.logger_mode) {
		case NFQUEUE_LOGGER_MODE_LOGTOFILE:
		case NFQUEUE_LOGGER_MODE_LOGTOUDP:
			nfqueue_logger_cb = (nfqueue_logger_cb_t)&nfqueue_logger_log_packet;
			break;

		case NFQUEUE_LOGGER_MODE_HTTP_REDIRECT:
			nfqueue_logger_cb = (nfqueue_logger_cb_t)&nfqueue_logger_http_redirect;
			break;

		default:
			fprintf(stderr, "%s: no call-back is identified for mode %d, exit\n", __FUNCTION__, nfqueue_logger_cb_data.logger_mode);
			return;
	}

	/* opening nfqueue handler */
	h = nfq_open();
	if (!h) {
		fprintf(stderr, "%s: error in nfq_open() for queue %d\n", __FUNCTION__, nfqueue_logger_cb_data.nfqueue_num);
		return;
	}

doBind:
	/* unbinding existing nf_queue handler for AF_INET (if any) */
	rv = nfq_unbind_pf(h, AF_INET);
	/* we don't need the following error handling with kernel 2.6.23 or above */
#if 0
	if (rv < 0) {
		fprintf(stderr, "%s: error in nfq_unbind_pf()\n");
	}
#endif
	
	/* binds the given queue connection handle to process packets */
	rv = nfq_bind_pf(h, AF_INET);
	if (rv < 0) {
		fprintf(stderr, "%s: error in nfq_bind_pf() for queue %d\n", __FUNCTION__, nfqueue_logger_cb_data.nfqueue_num);
		if (!bindAgain) { /* hmm...race condition? try again */
			bindAgain = 1;
			goto doBind;
		}
		fprintf(stderr, "%s: error in nfq_bind_pf() for queue %d always\n", __FUNCTION__, nfqueue_logger_cb_data.nfqueue_num);
		nfq_close(h);
		return;
	}
	NFQUEUE_LOGGER_DEBUG("%s: bind to queue %d, mode %d", __FUNCTION__, nfqueue_logger_cb_data.nfqueue_num, nfqueue_logger_cb_data.logger_mode);

	/* create the queue */
	qh = nfq_create_queue(h, nfqueue_logger_cb_data.nfqueue_num, nfqueue_logger_cb, &nfqueue_logger_cb_data);
	if (!qh) {
		fprintf(stderr, "%s: error in nfq_create_queue() for queue %d\n", __FUNCTION__, nfqueue_logger_cb_data.nfqueue_num);
		nfq_unbind_pf(h, AF_INET);
		nfq_close(h);
		return;
	}

	/* sets the amount of data to be copied to userspace 
	 * for each packet queued to the given queue 
	 */
	rv = nfq_set_mode(qh, NFQNL_COPY_PACKET, nfqueue_logger_cb_data.interested_len);
	if (rv < 0) {
		fprintf(stderr, "%s: error in nfq_set_mode() for queue %d\n", __FUNCTION__, nfqueue_logger_cb_data.nfqueue_num);
		nfq_unbind_pf(h, AF_INET);
		nfq_destroy_queue(qh);
		nfq_close(h);
		return;
	}

	/* fetchs the netlink handle associated with the given queue connection handle
	 * it would be useful if you wish to perform other netlink communication directly
	 * after opening a queue without opening a new netlink connection
	 */
	nh = nfq_nfnlh(h);

	/* fetchs the file descruptor for the netlink connection assocaited with the 
	 * given queue connection handle. it will be used later for receiving the queued
	 * packets
	 */
	fd = nfnl_fd(nh);

	buf_size = nfqueue_logger_cb_data.interested_len * 2;
	buf = (char *)malloc(buf_size);

	if (!buf) {
		fprintf(stderr, "%s: error in allocating buffer for queue %d\n", __FUNCTION__, nfqueue_logger_cb_data.nfqueue_num);
		nfq_unbind_pf(h, AF_INET);
		nfq_destroy_queue(qh);
		nfq_close(h);
		return;
	}

	wait_time.tv_sec = 2;
	wait_time.tv_usec = 0;
	while (!nfqueue_logger_exit) {
		FD_ZERO(&rfds);
		FD_SET(fd, &rfds);

		n = select(fd + 1, &rfds, NULL, NULL, NULL);
		if (n <= 0)
			continue;

		if (!FD_ISSET(fd, &rfds))
			continue;

		do {
			n = recv(fd, buf, buf_size, 0);
		} while (n < 0 && errno == EINTR);

		if (n < 0) {
			fprintf(stderr, "%s: error in receiving packet from queue %d\n", __FUNCTION__, nfqueue_logger_cb_data.nfqueue_num);
			nfq_destroy_queue(qh);
			nfq_close(h);
			return;
		} else if (n == 0) {
			fprintf(stderr, "%s: unable to receive packet from queue %d, possible empty queue?\n", __FUNCTION__, nfqueue_logger_cb_data.nfqueue_num);
		} else {
			NFQUEUE_LOGGER_DEBUG("------- received %d bytes from queue %d ----------", n, nfqueue_logger_cb_data.nfqueue_num);
			/* triggers an associated callback for the given packet received from the queue */
			nfq_handle_packet(h, buf, n);
		}
	}

#if 0 /* don't do unbind since it is systemwise, other applications using nf_queue will fail */
	NFQUEUE_LOGGER_DEBUG("%s: unbind to queue %d", __FUNCTION__, nfqueue_logger_cb_data.nfqueue_num);
	if (h)
		nfq_unbind_pf(h, AF_INET);
#endif
	if (qh)
		nfq_destroy_queue(qh);
	if (h)
		nfq_close(h);
}

static void nfqueue_logger_signal(int sig)
{
	if (sig == SIGTERM)
		nfqueue_logger_exit = 1;
}

static int nfqueue_logger_usage(void)
{
	printf("NFQUEUE-LOGGER v%s\n", NFQUEUE_LOGGER_VERSION);
	printf("  -d        :  enable debug mode\n");
	printf("  -q <qid>  :  NFQUEUE queue index\n");
	printf("  -l <flie> :  log to <file>\n");
	printf("  -r <URL>  :  http redirect to <URL>\n");
	printf("  -s <port> :  log to UDP socket <port>\n");
	printf("  -i <len>  :  interested packet length\n");

	return 0;
}

int main(int argc, char* argv[])
{
	int i;
	FILE *pid_fp;
	char pid_file[128];

	nfqueue_logger_cb_data.logger_mode = -1;
	nfqueue_logger_cb_data.nfqueue_num = -1;
	nfqueue_logger_cb_data.interested_len = NFQUEUE_LOGGER_LOG_SIZE_MAX;
	nfqueue_logger_cb_data.verdict = NF_ACCEPT;

	for(i = 1; i < argc; i++) {
		if (strcmp(argv[i], "-d") == 0) {
			nfqueue_logger_debug = 1;
		} else if (strcmp(argv[i], "-q") == 0) {
			if (++i == argc) {
				fprintf(stderr, "%s: -q missing queue index\n", argv[0]);
				exit(0);
			}
			nfqueue_logger_cb_data.nfqueue_num = (int)strtoul(argv[i], NULL, 10);
		} else if (strcmp(argv[i], "-l") == 0) {
			nfqueue_logger_cb_data.logger_mode = NFQUEUE_LOGGER_MODE_LOGTOFILE;
			if (++i == argc)
				sprintf(nfqueue_logger_target, NFQUEUE_LOGGER_LOG_FOLDER "/" NFQUEUE_LOGGER_LOG_FILE"-%d", getpid());
			else
				strcpy(nfqueue_logger_target, argv[i]);
			nfqueue_logger_cb_data.data = nfqueue_logger_target;
		} else if (strcmp(argv[i], "-s") == 0) {
			nfqueue_logger_cb_data.logger_mode = NFQUEUE_LOGGER_MODE_LOGTOUDP;
			if (++i == argc) {
				fprintf(stderr, "%s: -s missing socket port\n", argv[0]);
				exit(0);
			}
			nfqueue_logger_cb_data.data = (char *)strtoul(argv[i], NULL, 10);
		} else if (strcmp(argv[i], "-r") == 0) {
			nfqueue_logger_cb_data.logger_mode = NFQUEUE_LOGGER_MODE_HTTP_REDIRECT;
			if (++i == argc) {
				fprintf(stderr, "%s: -r missing URL\n", argv[0]);
				exit(0);
			}
			strcpy(nfqueue_logger_target, argv[i]);
			nfqueue_logger_cb_data.data = nfqueue_logger_target;
		} else if (strcmp(argv[i], "-i") == 0) {
			if (++i == argc) {
				fprintf(stderr, "%s: -i missing length\n", argv[0]);
				exit(0);
			}
			nfqueue_logger_cb_data.interested_len = (int)strtoul(argv[i], NULL, 10);
		} else if (strcmp(argv[i], "-v") == 0) {
			if (++i == argc) {
				fprintf(stderr, "%s: -v missing verdict parameter\n", argv[0]);
				exit(0);
			}
			nfqueue_logger_cb_data.verdict = strcmp(argv[i], "drop") == 0 ? NF_DROP : NF_ACCEPT;
		}
	}

	if (nfqueue_logger_cb_data.nfqueue_num < 0 ||
		nfqueue_logger_cb_data.logger_mode < 0)
		return nfqueue_logger_usage();


	sprintf(pid_file, "/var/run/nfqueue_logger-%d.pid", nfqueue_logger_cb_data.nfqueue_num);
	if (!(pid_fp = fopen(pid_file, "w"))) {
		perror(pid_file);
		return errno;
	}
	fprintf(pid_fp, "%d", getpid());
	fclose(pid_fp);

	signal(SIGPIPE, SIG_IGN);
	signal(SIGALRM, SIG_IGN);
	signal(SIGHUP, SIG_IGN);
	signal(SIGCHLD, SIG_IGN);
	signal(SIGINT, SIG_IGN);
	signal(SIGTERM, nfqueue_logger_signal);

	/* main loop */
	nfqueue_logger_netlink_loop();

	unlink(pid_file);

	return 0;
}

