/*
<:copyright-gpl 
 Copyright 2013 Arcadyan Technology 
 All Rights Reserved. 
 
 This program is free software; you can distribute it and/or modify it 
 under the terms of the GNU General Public License (Version 2) as 
 published by the Free Software Foundation. 
 
 This program is distributed in the hope it will be useful, but WITHOUT 
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License 
 for more details. 
 
 You should have received a copy of the GNU General Public License along 
 with this program; if not, write to the Free Software Foundation, Inc., 
 59 Temple Place - Suite 330, Boston MA 02111-1307, USA. 
:>
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>
#include <unistd.h>
#include <errno.h>
#include <net/if.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <linux/netfilter.h>
#include "libnetfilter_queue/libnetfilter_queue.h"
#include "nfqueue_logger.h"

#ifndef IFNAMESIZ
#define IFNAMESIZ 16
#endif

#define LOG_HEADER_LEN	(sizeof(int) + 6 + IFNAMESIZ)

static int log_sd = -1;
static struct sockaddr_in sd_to;
static int write_fd = -1;
static char *write_buf = NULL;
static int write_buf_len = 0;

/*
 * log format:
 * lenght of log(sizeof(int)), source MAC(6), incoming port(IF_NAMESIZE), packet data(?)
*/

/* the return error codes for nfnetlink_queue are:
 * 0 => ok
 * >0 => soft error
 * <0 => hard error
 */

static int nfqueue_logger_log_packet_update_write_buf(struct nfq_q_handle *qh, struct nfq_data *nfq_pkt, void *data)
{
	int rv = 0;
	nfqueue_logger_cb_data_t *nfqueue_logger = (nfqueue_logger_cb_data_t *)data;
	struct nfqnl_msg_packet_hdr *pktHdr;
	struct nfqnl_msg_packet_hw *pktHw;
	int id = -1, pktLen;
	unsigned char *payload;
	char *cp;

	pktHdr = nfq_get_msg_packet_hdr(nfq_pkt);
	if (!pktHdr) {
		fprintf(stderr, "%s: unable to get packet header for queue %d\n", __FUNCTION__, nfqueue_logger->nfqueue_num);
		return -1;
	}
	
	id = ntohl(pktHdr->packet_id);

	pktLen = nfq_get_payload(nfq_pkt, &payload);
	if (pktLen < 0) {
		fprintf(stderr, "%s: unable to get packet payload for queue %d\n", __FUNCTION__, nfqueue_logger->nfqueue_num);
		/* ops, we can not handle this packet, still forward it anyway */
		return -1;
	}

	NFQUEUE_LOGGER_DEBUG("len = %d, protocol = 0x%04x, hook = %u, id = %u", pktLen, ntohs(pktHdr->hw_protocol), pktHdr->hook, id);
	if (nfqueue_logger_debug)
		nfqueue_logger_dump_payload((char *)payload);

	/* the hardware address is only available at certain hook points
	 * it means the pktHw might be a NULL pointer
	 */
	pktHw = nfq_get_packet_hw(nfq_pkt);

	
	if (!write_buf) {
		write_buf_len = nfqueue_logger->interested_len + LOG_HEADER_LEN;
		write_buf = (char *)malloc(write_buf_len);
		if (!write_buf) {
			fprintf(stderr, "%s: unable to allocate write buffer for queue %d\n", __FUNCTION__, nfqueue_logger->nfqueue_num);
			return -1;
		}
	}

	memset(write_buf, 0, write_buf_len);
	
	cp = write_buf;
	
	pktLen = pktLen > nfqueue_logger->interested_len ? nfqueue_logger->interested_len : pktLen;

	/* skip the first four bytes, will be updated later */
	cp += sizeof(int);

	/* MAC address */
	if (pktHw)
		memcpy(cp, pktHw->hw_addr, 6);
	else
		memset(cp, 0, 6);
	cp += 6;

	/* ingress physical interface */
	if (nfqueue_logger_query_physindev(nfq_pkt, cp) != 0)
		memset(cp, 0, IFNAMESIZ);
	cp += IFNAMESIZ;

	/* packet data */
	memcpy(cp, payload, pktLen);

	pktLen += LOG_HEADER_LEN;
	memcpy(write_buf, &pktLen, sizeof(int));

	if (id >= 0) {
		rv = nfq_set_verdict(qh, id, nfqueue_logger->verdict, 0, NULL);
		if (rv == -1) {
			fprintf(stderr, "%s: unable to set verdict\n", __FUNCTION__);
			return -1;
		}
	}

	return pktLen;
}

static int nfqueue_logger_log_packet_to_file(struct nfq_q_handle *qh, struct nfgenmsg *nfmsg, struct nfq_data *nfq_pkt, void *data)
{
	int write_len;
	nfqueue_logger_cb_data_t *nfqueue_logger = (nfqueue_logger_cb_data_t *)data;

	if (write_fd < 0) {
		write_fd = open(nfqueue_logger->data, O_WRONLY);
		if (write_fd < 0) {
			fprintf(stderr, "%s: unable to open target fifo %s for queue %d\n", __FUNCTION__, nfqueue_logger->data, nfqueue_logger->nfqueue_num);
			return -1;
		}
	}

	write_len = nfqueue_logger_log_packet_update_write_buf(qh, nfq_pkt, data);

	if (write_len > 0)
		write(write_fd, write_buf, write_len);

	return 0;
}

static int nfqueue_logger_log_packet_to_udp(struct nfq_q_handle *qh, struct nfgenmsg *nfmsg, struct nfq_data *nfq_pkt, void *data)
{
	int write_len = 0;
	nfqueue_logger_cb_data_t *nfqueue_logger = (nfqueue_logger_cb_data_t *)data;

	if (log_sd < 0) {
		int sin_port = (int)nfqueue_logger->data;
		
		memset(&sd_to, 0, sizeof(sd_to));
		sd_to.sin_family = AF_INET;
		sd_to.sin_addr.s_addr = htonl(INADDR_LOOPBACK);
		sd_to.sin_port = htons(sin_port);

		log_sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
		if (log_sd < 0) {
			fprintf(stderr, "%s: unable to create socket %d for queue %d\n", __FUNCTION__, sin_port, nfqueue_logger->nfqueue_num);
			return -1;
		}
	}

	write_len = nfqueue_logger_log_packet_update_write_buf(qh, nfq_pkt, data);

	if (write_len > 0)
		sendto(log_sd, write_buf, write_len, 
									0, (struct sockaddr *)&sd_to, sizeof(struct sockaddr_in));

	return 0;
}

int nfqueue_logger_log_packet(struct nfq_q_handle *qh, struct nfgenmsg *nfmsg, struct nfq_data *nfq_pkt, void *data)
{
	nfqueue_logger_cb_data_t *nfqueue_logger = (nfqueue_logger_cb_data_t *)data;

	switch (nfqueue_logger->logger_mode) {
		case NFQUEUE_LOGGER_MODE_LOGTOFILE:
			return nfqueue_logger_log_packet_to_file(qh, nfmsg, nfq_pkt, data);

		case NFQUEUE_LOGGER_MODE_LOGTOUDP:
			return nfqueue_logger_log_packet_to_udp(qh, nfmsg, nfq_pkt, data);

		default:
			return -1;
	}

	return 0;
}

