/* Copyright (c) 2004-2009 Piotr 'QuakeR' Gasidlo <quaker@barbara.eu.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <xtables.h>
#include <string.h>
#include <getopt.h>

#include <linux/netfilter_ipv4/ipt_account.h>
        
static void account_help(void) {
  printf(
      "account v%s options:\n"
      "--aaddr network/netmask\n"
      " defines network/netmask for which make statistics.\n"
      "--aname name\n"
      " defines name of list where statistics will be kept. If no is\n"
      " specified DEFAULT will be used.\n"
      "--ashort\n"
      "       table will colect only short statistics (only total counters\n"
      "       without splitting it into protocols.\n"
  , 
  XTABLES_VERSION
  );
};

static struct option account_opts[] = {
  { "aaddr",  1, NULL, '1' },
  { "aname",  1, NULL, '2' },
  { "ashort", 0, NULL, '3' },
  { .name = NULL }
};

/* Helper functions for parse_network */
int parseip(const char *parameter, u_int32_t *ip) {
  
  char buffer[16], *bufferptr, *dot;
  unsigned int i, shift, part;

  if (strlen(parameter) > 15)
    return 0;

  strncpy(buffer, parameter, 15);
  buffer[15] = 0;

  bufferptr = buffer;

  for (i = 0, shift = 24, *ip = 0; i < 3; i++, shift -= 8) {
    /* no dot */
    if ((dot = strchr(bufferptr, '.')) == NULL)
      return 0;
    /* not a number */
    if ((part = strtol(bufferptr, (char**)NULL, 10)) < 0) 
      return 0; 
    /* to big number */
    if (part > 255)
      return 0;
    *ip |= part << shift;   
    bufferptr = dot + 1;
  }
  /* not a number */
  if ((part = strtol(bufferptr, (char**)NULL, 10)) < 0) 
    return 0;
  /* to big number */
  if (part > 255)
    return 0;
  *ip |= part;
  return 1;
}

static void parsenetwork(const char *parameter, u_int32_t *network) {
  if (!parseip(parameter, network))
    xtables_error(PARAMETER_PROBLEM, "account: wrong ip in network");
}

static void parsenetmaskasbits(const char *parameter, u_int32_t *netmask) {
  
  u_int32_t bits;
  
  if ((bits = strtol(parameter, (char **)NULL, 10)) < 0 || bits > 32)
    xtables_error(PARAMETER_PROBLEM, "account: wrong netmask");

  *netmask = 0xffffffff << (32 - bits);
}

static void parsenetmaskasip(const char *parameter, u_int32_t *netmask) {
  if (!parseip(parameter, netmask))
    xtables_error(PARAMETER_PROBLEM, "account: wrong ip in netmask");
}

static void parsenetmask(const char *parameter, u_int32_t *netmask) 
{
  if (strchr(parameter, '.') != NULL)
    parsenetmaskasip(parameter, netmask);
  else
    parsenetmaskasbits(parameter, netmask);
}

static void parsenetworkandnetmask(const char *parameter, u_int32_t *network, u_int32_t *netmask) 
{
  
  char buffer[32], *slash;

  if (strlen(parameter) > 31)
    /* text is to long, even for 255.255.255.255/255.255.255.255 */
    xtables_error(PARAMETER_PROBLEM, "account: wrong network/netmask");

  strncpy(buffer, parameter, 31);
  buffer[31] = 0;

  /* check whether netmask is given */
  if ((slash = strchr(buffer, '/')) != NULL) {
    parsenetmask(slash + 1, netmask);
    *slash = 0;
  } else
    *netmask = 0xffffffff;
  parsenetwork(buffer, network);

  if ((*network & *netmask) != *network)
    xtables_error(PARAMETER_PROBLEM, "account: wrong network/netmask");
}


/* Function gets network & netmask from argument after --aaddr */
static void parse_network(const char *parameter, struct t_ipt_account_info *info) {

  parsenetworkandnetmask(parameter, &info->network, &info->netmask);
  
}

/* validate netmask */
inline int valid_netmask(u_int32_t netmask) {
  while (netmask & 0x80000000)
    netmask <<= 1;
  if (netmask != 0)
    return 0;
        return 1;
}

/* validate network/netmask pair */
inline int valid_network_and_netmask(struct t_ipt_account_info *info) {
  if (!valid_netmask(info->netmask))
    return 0;
  if ((info->network & info->netmask) != info->network)
    return 0;
  return 1;
}



/* Function initializes match */
static void account_init(struct xt_entry_match *match) 
{
  struct t_ipt_account_info *info = (struct t_ipt_account_info *)(match)->data;


  /* set default table name to DEFAULT */
  strncpy(info->name, "DEFAULT", IPT_ACCOUNT_NAME_LEN);
  info->shortlisting = 0;
  info->table = NULL;
  
}

/* Function parses match's arguments */
static int account_parse(int c, 
    char **argv, 
    int invert, 
    unsigned int *flags,
    const void *entry,
    struct xt_entry_match **match
    )
{ 
  struct t_ipt_account_info *info = (struct t_ipt_account_info *)(*match)->data;

  switch (c) {
    
    /* --aaddr */
    case '1':
      parse_network(optarg, info);
      if (!valid_network_and_netmask(info))
        xtables_error(PARAMETER_PROBLEM, "account: wrong network/netmask");
      *flags = 1;
      break;
      
    /* --aname */
    case '2':
      if (strlen(optarg) < IPT_ACCOUNT_NAME_LEN) {
        strncpy(info->name, optarg, IPT_ACCOUNT_NAME_LEN);
        info->name[IPT_ACCOUNT_NAME_LEN] = '\0';
      } else
        xtables_error(PARAMETER_PROBLEM, "account: Too long table name");      
      break;  
    /* --ashort */
    case '3':
      info->shortlisting = 1;
      break;
    default:
      return 0;     
  }
  return 1; 
}

/* Final check whether network/netmask was specified */
static void account_final_check(unsigned int flags) {
  if (!flags)
    xtables_error(PARAMETER_PROBLEM, "account: You need specify '--aaddr' parameter");
}

/* Function used for printing rule with account match for iptables -L */
static void account_print(
    const void *ip, 
    const struct xt_entry_match *match,
    int numeric) {
  
  struct t_ipt_account_info *info = (struct t_ipt_account_info *)match->data;
  char network_str[INET_ADDRSTRLEN];
  char netmask_str[INET_ADDRSTRLEN];
  struct in_addr addr;
  
  printf("account: ");
  printf("network/netmask: ");

  addr.s_addr = htonl(info->network);
  inet_ntop(AF_INET, &addr, network_str, sizeof(network_str));
  addr.s_addr = htonl(info->netmask);
  inet_ntop(AF_INET, &addr, netmask_str, sizeof(netmask_str));
  printf("%s/%s ", network_str, netmask_str);
  
  printf("name: %s ", info->name);
  if (info->shortlisting)
    printf("short-listing ");
}

/* Function used for saving rule containing account match */
static void account_save(
    const void *ip,
    const struct xt_entry_match *match
) {

  struct t_ipt_account_info *info = (struct t_ipt_account_info *)match->data;
  char network_str[INET_ADDRSTRLEN];
  char netmask_str[INET_ADDRSTRLEN];
  struct in_addr addr;
  
  printf("--aaddr ");

  addr.s_addr = htonl(info->network);
  inet_ntop(AF_INET, &addr, network_str, sizeof(network_str));
  addr.s_addr = htonl(info->netmask);
  inet_ntop(AF_INET, &addr, netmask_str, sizeof(netmask_str));
  printf("%s/%s ", network_str, netmask_str);
  
  printf("--aname %s ", info->name);
  if (info->shortlisting)
    printf("--ashort ");
}

static struct xtables_match account = { NULL,
  .name = "account",
  .version = XTABLES_VERSION,
  .family = NFPROTO_UNSPEC,
  .size = XT_ALIGN(sizeof(struct t_ipt_account_info)),
  .userspacesize = XT_ALIGN(sizeof(struct t_ipt_account_info)),
  .help = account_help,
  .init = account_init,
  .parse = account_parse,
  .final_check = account_final_check,
  .print = account_print,
  .save = account_save,
  .extra_opts = account_opts
};

/* Function which registers match */
void _init(void)
{
  xtables_register_match(&account);
}
  
