/*
 * Mini pidofproc implementation for busybox.
 *
 * Copyright (C) 2005 by David Seikel won_fang@yahoo.com.au
 *
 * Clean room implementation of LSB init.d specs.
 * I didn't steal any of this, honest B-).
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA
 *
 */
 
//usage:#define pidofproc_trivial_usage
//usage:       "pidofproc [-p pidfile] pathname\n"
//usage:#define pidofproc_full_usage "\n\n"
//usage:       "Returns one or more pid(s) for a particular daemon.\n"
//usage:		"\nOptions:"
//usage:		"\n        -p pidfile      path to pidfile."
//usage:#define pidofproc_example_usage
//usage:       "\n"

#include <getopt.h>
#include <string.h>
#include <unistd.h>

#include "busybox.h"
#include "init_d.h"

int checkpid(char *pid)
{
	int found = 0;

	RESERVE_CONFIG_BUFFER(proc, PATH_MAX);

	snprintf(proc, PATH_MAX, "/proc/%s", pid);
	if (stat(proc, &path_stat) == 0) {
		char *state = NULL;

		snprintf(proc, PATH_MAX, "/proc/%s/stat", pid);
		state = quick_read(proc);
		if (state != NULL) {
			/* Read and interpret the third space seperated field (State - DRSW = OK - ZX = DEAD_PID - T = STOPPED) */
			if (state[0] != '\0') {
				while ((*state != ' ') && (*state != '\0'))
					state++;
				while (*state == ' ')
					state++;
				while ((*state != ' ') && (*state != '\0'))
					state++
// Should compare the name while scanning this.
						;
				while (*state == ' ')
					state++;
				switch (*state) {
				case 'D':
				case 'R':
				case 'S':
				case 'W':
				case 'T':	/* Until proven otherwise, I'm assuming this means OK. */
					found = 1;
				}
			}
		} else {		/* Paranoid fallbacks. */

			snprintf(proc, PATH_MAX, "/proc/%s/exe", pid);
			if (stat(proc, &path_stat) == 0) {
// Should check that it is a link to our executable.
				found = 1;
			} else {
				snprintf(proc, PATH_MAX, "/proc/%s/cmdline", pid);
				if (stat(proc, &path_stat) == 0) {
// Should compare the name.
					found = 1;
				}
			}
		}

	}

	RELEASE_CONFIG_BUFFER(proc);
	return found;
}


int pidofproc(char *pidfile, char *pathname, char **pids)
{
	int status = PIDOFPROC_STATUS_UNKNOWN;
	char *our_pidfile = pidfile;
	char *our_pids = NULL;

	if (our_pidfile == NULL)
		xasprintf(&our_pidfile, "/var/run/%s.pid",
					 bb_get_last_path_component_strip(pathname));

	our_pids = quick_read(our_pidfile);

	if (our_pids != NULL) {
		int i;
		char *pid;
		char *strtok_temp;

		status = PIDOFPROC_STATUS_DEAD_PID;
		if (pids != NULL) {
			(*pids) = (char *) xmalloc(sizeof(char) * strlen(our_pids) + 1);
			(*pids)[0] = '\0';
		}

		pid = strtok_r(our_pids, " \n\r", &strtok_temp);
		for (i = 0; pid != NULL; i++) {
			if (checkpid(pid)) {
				if (pids != NULL) {
					strcat(*pids, pid);
					strcat(*pids, " ");
				}
				status = PIDOFPROC_STATUS_OK;
			}

			pid = strtok_r(NULL, " \n\r", &strtok_temp);
		}

		free(our_pids);
	} else
		status = PIDOFPROC_STATUS_NOT_RUNNING;

	if (pidfile == NULL)
		free(our_pidfile);

	return status;
}


/*
pidofproc [-p pidfile] pathname
    This function returns one or more pid(s) for a particular
    daemon using the algorithm given above. Only pids of running processes
    should be returned.

    Multiple pid(s) shall be separated by a single space in the pidfile
    and in the output of pidofproc.

    Compliant applications may use the basename instead of the pathname.
    pidofproc should return the LSB defined exit status
    codes for "status". It shall return 0 if the program is
    running and not 0 otherwise.
*/

int pidofproc_main(int argc, char **argv)
{
	int result = EXIT_FAILURE;
	char *pidfile = NULL;
	char *pids = NULL;
	int opt;

	while ((opt = getopt(argc, argv, "p:")) > 0) {
		switch (opt) {
		case 'p':
			pidfile = optarg;
			break;

		default:
			bb_show_usage();
		}
	}

	/* We require exactly one argument: the path name */
	if (optind != (argc - 1))
		bb_show_usage();

	result = pidofproc(pidfile, argv[optind], &pids);
	if (pids != NULL)
		printf("%s\n", pids);

	return result;
}
