/*
 * Mini killproc implementation for busybox.
 *
 * Copyright (C) 2005 by David Seikel won_fang@yahoo.com.au
 *
 *
 * Clean room implementation of LSB init.d specs.
 * A tiny bit was stolen from procps/kill.c in busybox. 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA
 *
 */
 
//usage:#define killproc_trivial_usage
//usage:       "killproc [-p pidfile] [-t seconds] pathname [signal]\n"
//usage:#define killproc_full_usage "\n\n"
//usage:       "Kill all running processes of a named program.\n"
//usage:		"\nOptions:"
//usage:		"\n        -p pidfile      path to pidfile."
//usage:		"\n        -t seconds      timeour for waiting kill processes."
//usage:#define killproc_example_usage
//usage:       "\n"

#include <ctype.h>
#include <signal.h>
#include <string.h>
#include <unistd.h>

#include "busybox.h"

int killproc(char *pidfile, char *pathname, int my_signal, int wait_to)
{
	int pid_num;
	char *pid = NULL, *pids = NULL;
	char *strtok_temp;
	int status = pidofproc(pidfile, pathname, &pids);
	int partsec = 5 * wait_to; /* We look 5 times within a second */
	
	if (status == PIDOFPROC_STATUS_OK) {
again:
		while ((pid = strtok_r(pids, "", &strtok_temp)) != NULL) {
			pids = NULL;
			if (!isdigit(*pid)) {
				bb_error_msg("Bad PID '%s'", pid);
				continue;
			}
			pid_num = strtol(pid, NULL, 0);
			
			if (kill(pid_num, my_signal) != 0)
				bb_perror_msg("Could not kill pid '%d'", pid_num);
		}
		
		usleep(60*1000);	/* 60 ms time for the process and its childs */
		
		status = pidofproc(pidfile, pathname, &pids);
		if ((status == PIDOFPROC_STATUS_OK) && (partsec-- > 0)) {
			usleep(2 * 100 * 1000); /* sleep 0.2 seconds and try again */
			goto again;
		}
	}
	
	if (partsec == 0)
		bb_perror_msg("Could not kill '%s' due to timeout", pathname);
		
	if (status == PIDOFPROC_STATUS_DEAD_PID 
		|| status == PIDOFPROC_STATUS_NOT_RUNNING) {
		if (pidfile != NULL)
			remove_file(pidfile, FILEUTILS_FORCE);
		status = PIDOFPROC_STATUS_OK;
	}

	return status;
}


/*
killproc [-p pidfile] [-t seconds] pathname [signal]
    This stops the specified program. The program is 
    found using the algorithm given above. If a
    signal is specified, using the
    -signal_name or -signal_number syntaxes
    as specified by the kill command,
    the program is sent that signal.
    Otherwise, a SIGTERM followed by a SIGKILL
    after some number of seconds shall be sent.
    If a program has been terminated, the pidfile should be removed if the
    terminated process has not already done so.
    Compliant applications may use the basename instead of the 
    pathname.

    killproc should return the LSB defined exit status codes. If called
    without a signal, it shall return 0 if the program has been stopped or
    is not running and not 0 otherwise. If a signal is given, it shall return 0
    only if the program is running.
*/


	/* getopt not used, because I don't think it can handle [signal] */
int killproc_main(int argc, char **argv)
{
	int i;
	int my_signal = 0;
	int wait_to = 0;
	char *pidfile = NULL;
	char *pathname = NULL;
	char *signame = NULL;

	for (i = 1; i < argc; i++) {
		char *p = argv[i];

		if (*p == '-') {
			p++;
			if (*p == 'p') {
				if ((i + 1) < argc)
					pidfile = argv[++i];
				else
					bb_show_usage();
			} else if (*p == 't') {
				if ((i + 1) < argc)
					wait_to = atoi(argv[++i]);
				else
					bb_show_usage();
			} else
				signame = &argv[i][1];
		} else if (pathname == NULL)
			pathname = p;
		else if (signame == NULL)
			signame = p;
		else
			bb_show_usage();
	}

	if (pathname == NULL)
		bb_show_usage();

	if (signame != NULL) {
		my_signal = get_signum(signame);
		if (my_signal == 0) {
			bb_error_msg("unknown signal '%s'", signame);
			bb_show_usage();
		}
	} else {
		my_signal = SIGTERM;
	}
	
	return killproc(pidfile, pathname, my_signal, wait_to);
}
