/*
 * Copyright (C) 2004 by David Seikel won_fang@yahoo.com.au
 *
 * llist_add_to_end() lifted whole from busybox networking/ifupdown.c, 
 * so it should probably be in libbb.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA
 *
 */

#include <dirent.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <string.h>
#include <sys/poll.h>
#include <sys/wait.h>
#include <sys/utsname.h>	/* for uname(2) */

#include "busybox.h"
#include "init_d.h"


const static nodes_t nodes[] = {
	{"hda", S_IFBLK, 3, 0, 0},
	{"hdb", S_IFBLK, 3, 64, 0},
	{"hdc", S_IFBLK, 22, 0, 0},
	{"hdd", S_IFBLK, 22, 64, 0},
	{"ram", S_IFBLK, 1, 0, 9},
	{"fd", S_IFBLK, 2, 0, 1},
	{"loop", S_IFBLK, 7, 0, 63},
	{"cloop", S_IFBLK, 240, 0, 7},
	{"vcs", S_IFBLK, 7, 0, 9},
	{"vcsa", S_IFBLK, 7, 0, 9},
	{0, 0, 0, 0, 0}
};


static char *shell = 0;
struct stat path_stat;


char *argv_cat(int argc, char **argv)
{
	int i;
	int length = 1;
	char *message = NULL;

	for (i = 0; i < argc; i++)
		length += strlen(argv[i]) + 1;
	message = (char *) xmalloc(sizeof(char) * length);

	if (message != NULL) {
		message[0] = '\0';
		for (i = 1; i < argc; i++) {
			/* These strcat's are not dangerous, since we checked sizes above. */
			strcat(message, argv[i]);
			if (i < (argc - 1))
				strcat(message, " ");
		}
	}
	return message;
}


char *big_chomp(char *s)
{
	char *lc = index(s, '\n');

	if (lc)
		*lc = '\0';
	return s;
}

#if 0
/* I found out that networking/ifupdown.c has a similar function
 * with an identical name AFTER I wrote this.
 */
char *doit(int mode, char *command, ...)
{
	int buffsize = 63;
	char *buffer = NULL;
	int dataPipe[2] = { -1, -1 };
	int statusPipe[2] = { -1, -1 };
	int n;
	pid_t pid = 0;
	volatile int vfork_exec_errno = 0;
	char **args;
	char *commandBuffer;

	if (mode & DAEMON)
		mode |= FORK | QUIET;

	va_list p;
	int r;

	va_start(p, command);
	r = vasprintf(&commandBuffer, command, p);
	va_end(p);

	if (r < 0)
		bb_perror_msg_and_die("doit");

	if (shell == 0) {
		shell = getenv("SHELL");
		if (shell == 0)
			shell = "/bin/sh";
	}

	args = (char **) xmalloc(sizeof(char *) * 4);
	n = 0;
	args[n++] = shell;
	args[n++] = "-c";
	if (mode & QUIET)
		bb_xasprintf(&(args[n++]), "%s 2>/dev/null", commandBuffer);
	else
		args[n++] = commandBuffer;
	args[n++] = 0;

	if (!(mode & NOFORK)) {
		if (!(mode & DAEMON)) {
			if (pipe(dataPipe) < 0 || pipe(statusPipe) < 0)
				bb_perror_msg("Failed to create pipe");
			signal(SIGPIPE, SIG_IGN);	/* we only want EPIPE on errors */
		}
//  pid = vfork();
		pid = fork();
	}
	if (pid == 0) {		/* child */
		if ((!(mode & NOFORK)) && (!(mode & NOFORK))) {
			close(STDIN_FILENO);
			dup2(dataPipe[1], STDOUT_FILENO);
			dup2(dataPipe[1], STDERR_FILENO);
			close(dataPipe[0]);
			close(statusPipe[0]);
			fcntl(statusPipe[1], F_SETFD, FD_CLOEXEC);	/* close on exec shows sucess */
		}

		if ((mode & DAEMON)) {
			if (!(mode & NOFORK)) {
				close(dataPipe[1]);
				close(statusPipe[1]);
				close(STDOUT_FILENO);
				close(STDERR_FILENO);
			}
			daemon(1, 0);
		}
		errno = 0;
		execvp(shell, (char **) args);

		vfork_exec_errno = errno;
		if (!(mode & NOFORK))
			close(statusPipe[1]);
		_exit(-1);
	} else if (pid > 0) {	/* parent */
		close(dataPipe[1]);
		close(statusPipe[1]);

		while (1) {
			char buf;

			n = read(statusPipe[0], &buf, 1);

			if ((n < 0) && ((errno == EAGAIN) || (errno == EINTR))
				&& !(mode & FORK))
				continue;	/* try it again */
			else if (n == 0 && vfork_exec_errno != 0) {
				errno = vfork_exec_errno;
				bb_perror_msg("Could not exec process");
			}
			break;
		}
		close(statusPipe[0]);

		if (buffer == NULL)
			buffer = (char *) xcalloc(buffsize + 1, sizeof(char));

		if ((mode & FORK) == 0) {
			ssize_t cc;
			struct pollfd poller[1];

			poller[0].fd = dataPipe[0];
			poller[0].events = POLLIN | POLLPRI;
			poller[0].revents = 0;

			n = 0;
			while (1) {
				if ((buffsize - n) <= 0) {
					int i;

					buffsize += buffsize;
					buffer = xrealloc(buffer, sizeof(char) * (buffsize + 1));
					for (i = n; i <= buffsize; i++)
						buffer[i] = '\0';
				}

				errno = 0;
				cc = poll(poller, 1, 1010);	/* network sleeps for 1000 between '.' outputs */
				if (cc > 0)
					cc = read(dataPipe[0], &buffer[n], buffsize - n);

				if (cc < 0) {
					if (errno == EINTR)
						continue;
					if (errno == EAGAIN)
						continue;
					break;
				} else if (cc == 0)
					break;

				if ((mode & (QUIET | REDIR)) == 0)
					bb_printf(&buffer[n]);
				n += cc;
			}
		}

		if (mode & REDIR)
			chomp(buffer);
	} else {
		bb_perror_msg("Failed to fork process");
	}

	n = 0;
	if (pid) {
		close(dataPipe[0]);

		if ((mode & FORK) == 0) {
			if (waitpid(pid, &n, 0) == -1)
				bb_printf("Couldn't wait?");
		}
		if (WIFEXITED(n))
			n = WEXITSTATUS(n);
		else
			n = -1;
	}

	free(args);
	free(commandBuffer);
	errno = n;
	return buffer;
}
#endif

llist_t *llist_delete(llist_t ** head, llist_t * previous, llist_t * current)
{
	if (previous == NULL) {
		*head = current->link;
		free(current);
		current = *head;
	} else {
		previous->link = current->link;
		free(current);
		current = previous->link;
	}

	return current;
}


void make_disk(char *token)
{
	int i;

	RESERVE_CONFIG_BUFFER(temp, PATH_MAX);

	if (nodes == NULL)
		return;

	for (i = 0; nodes[i].name != NULL; i++) {
		if (strncmp(nodes[i].name, token, 3) == 0) {
			int m = atoi(&token[3]);

			snprintf(temp, PATH_MAX, "/dev/%s", token);
			mknod(temp, nodes[i].mode,
				  makedev(nodes[i].major, nodes[i].minor + m));
			break;
		}
	}
	snprintf(temp, PATH_MAX, "/media/%s", token);
	bb_make_directory(temp, -1l, FILEUTILS_RECUR);
	RELEASE_CONFIG_BUFFER(temp);
}

#if 0
void make_ram_disk(int size, int number, char *place, int TMPFS)
{
	if (size > 0) {
		char *place2;

		bb_printf("Creating ramdisk at %s\n", place);
		bb_xasprintf(&place2, "%s2", place);
		if (copy_file
			(place, place2, FILEUTILS_RECUR | FILEUTILS_PRESERVE_STATUS) >= 0)
			remove_file(place, FILEUTILS_RECUR | FILEUTILS_FORCE);
		bb_make_directory(place, -1l, 0);
		if (TMPFS)
			quick_mount("tmpfs", "/dev/null", place, "-n -o size=%i",
						size * 1024);
		else {
			doit(QUIET, "mkfs.minix /dev/ram%i %i", number, size);
			quick_mount("minix", "", "", "/dev/ram%i %s", number, place);
		}
		if (copy_file
			(place2, place, FILEUTILS_RECUR | FILEUTILS_PRESERVE_STATUS) >= 0)
			remove_file(place2, FILEUTILS_RECUR | FILEUTILS_FORCE);
		free(place2);
	}
}


void quick_mount(char *type, char *device, char *path, char *data, ...)
{
	char *dataBuffer;
	va_list p;
	int r;

	va_start(p, data);
	r = vasprintf(&dataBuffer, data, p);
	va_end(p);
	if (r < 0)
		bb_perror_msg("quick_mount");
	else {
		doit(QUIET, "busybox mount -t %s %s %s %s", type, device, path,
			 dataBuffer);
	}

	free(dataBuffer);
}
#endif


char *quick_read(char *filename)
{
	char *result = NULL;

	if (stat(filename, &path_stat) == 0) {
//  if (S_ISREG(path_stat.st_mode))
		{
			int ifd;
			ssize_t size = path_stat.st_size;

			if (size <= 0)
				size = 1024;
			result = (char *) xmalloc(sizeof(char) * size + 1);
			if ((ifd = open(filename, O_RDONLY)) < 0)
				bb_perror_msg("%s", filename);
			else {
				ssize_t total = full_read(ifd, result, size);

//      result[path_stat.st_size] = '\0';
				result[total] = '\0';
				if (close(ifd) < 0)
					bb_perror_msg("%s", filename);
			}
		}
	}

	return result;
}

#if 0
void quick_write(const char *filename, const char *data, ...)
{
	char *dataBuffer;
	va_list p;
	int r;

	va_start(p, data);
	r = vasprintf(&dataBuffer, data, p);
	va_end(p);
	if (r >= 0) {
		int ofd;

		if ((ofd = open(filename, O_WRONLY | O_CREAT | O_TRUNC, 0666)) < 0)
			bb_perror_msg("%s", filename);
		else {
			r = bb_full_write(ofd, dataBuffer, r);
			if (close(ofd) < 0)
				bb_perror_msg("%s", filename);
		}
	}
	if (r < 0)
		bb_perror_msg("quick_write");

	free(dataBuffer);
}
#endif
