/*
 * Copyright (C) 2000 Lennert Buytenhek
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <errno.h>
#include <asm/param.h>
#include "libbridge.h"
#include "brctl.h"

static int strtotimeval(struct timeval *tv, const char *time)
{
	double secs;
	if (sscanf(time, "%lf", &secs) != 1) 
		return -1;
	tv->tv_sec = secs;
	tv->tv_usec = 1000000 * (secs - tv->tv_sec);
	return 0;
}

static int br_cmd_addbr(int argc, char*const* argv)
{
	int err;

	switch (err = br_add_bridge(argv[1])) {
	case 0:
		return 0;

	case EEXIST:
		fprintf(stderr,	"device %s already exists; can't create "
			"bridge with the same name\n", argv[1]);
		return 1;
	default:
		fprintf(stderr, "add bridge failed: %s\n",
			strerror(err));
		return 1;
	}
}

static int br_cmd_delbr(int argc, char*const* argv)
{
	int err;

	switch (err = br_del_bridge(argv[1])){
	case 0:
		return 0;

	case ENXIO:
		fprintf(stderr, "bridge %s doesn't exist; can't delete it\n",
			argv[1]);
		return 1;

	case EBUSY:
		fprintf(stderr, "bridge %s is still up; can't delete it\n",
			argv[1]);
		return 1;

	default:
		fprintf(stderr, "can't delete bridge %s: %s\n",
			argv[1], strerror(err));
		return 1;
	}
}

static int br_cmd_addif(int argc, char *const* argv)
{
	const char *brname;
	int err;

	argc -= 2;
	brname = *++argv;

	while (argc-- > 0) {
		const char *ifname = *++argv;
		err = br_add_interface(brname, ifname);

		switch(err) {
		case 0:
			continue;

		case ENODEV:
			if (if_nametoindex(ifname) == 0)
				fprintf(stderr, "interface %s does not exist!\n", ifname);
			else
				fprintf(stderr, "bridge %s does not exist!\n", brname);
			break;

		case EBUSY:
			fprintf(stderr,	"device %s is already a member of a bridge; "
				"can't enslave it to bridge %s.\n", ifname,
				brname);
			break;

		case ELOOP:
			fprintf(stderr, "device %s is a bridge device itself; "
				"can't enslave a bridge device to a bridge device.\n",
				ifname);
			break;

		default:
			fprintf(stderr, "can't add %s to bridge %s: %s\n",
				ifname, brname, strerror(err));
		}
		return 1;
	}
	return 0;
}

static int br_cmd_delif(int argc, char *const* argv)
{
	const char *brname;
	int err;

	argc -= 2;
	brname = *++argv;

	while (argc-- > 0) {
		const char *ifname = *++argv;
		err = br_del_interface(brname, ifname);
		switch (err) {
		case 0:
			continue;

		case ENODEV:
			if (if_nametoindex(ifname) == 0)
				fprintf(stderr, "interface %s does not exist!\n", ifname);
			else
				fprintf(stderr, "bridge %s does not exist!\n", brname);
			break;

		case EINVAL:
			fprintf(stderr, "device %s is not a slave of %s\n",
				ifname, brname);
			break;

		default:
			fprintf(stderr, "can't delete %s from %s: %s\n",
				ifname, brname, strerror(err));
		}
		return 1;
	}
	return 0;
}

static int br_cmd_setageing(int argc, char *const* argv)
{
	int err;
	struct timeval tv;
	
	if (strtotimeval(&tv, argv[2])) {
		fprintf(stderr, "bad ageing time value\n");
		return 1;
	}

	err = br_set_ageing_time(argv[1], &tv);
	if (err)
		fprintf(stderr, "set ageing time failed: %s\n",
			strerror(err));

	return err != 0;
}

static int br_cmd_setbridgeprio(int argc, char *const* argv)
{
	int prio;
	int err;

	if (sscanf(argv[2], "%i", &prio) != 1) {
		fprintf(stderr,"bad priority\n");
		return 1;
	}

	err = br_set_bridge_priority(argv[1], prio);
	if (err)
		fprintf(stderr, "set bridge priority failed: %s\n",
			strerror(err));
	return err != 0;
}

static int br_cmd_setfd(int argc, char *const* argv)
{
	struct timeval tv;
	int err;

	if (strtotimeval(&tv, argv[2])) {
		fprintf(stderr, "bad forward delay value\n");
		return 1;
	}

	err = br_set_bridge_forward_delay(argv[1], &tv);
	if (err)
		fprintf(stderr, "set forward delay failed: %s\n",
			strerror(err));

	return err != 0;
}

static int br_cmd_sethello(int argc, char *const* argv)
{
	struct timeval tv;
	int err;

	if (strtotimeval(&tv, argv[2])) {
		fprintf(stderr, "bad hello timer value\n");
		return 1;
	}

	err = br_set_bridge_hello_time(argv[1], &tv);
	if (err)
		fprintf(stderr, "set hello timer failed: %s\n",
			strerror(err));

	return err != 0;
}

static int br_cmd_setmaxage(int argc, char *const* argv)
{
	struct timeval tv;
	int err;

	if (strtotimeval(&tv, argv[2])) {
		fprintf(stderr, "bad max age value\n");
		return 1;
	}
	err = br_set_bridge_max_age(argv[1], &tv);
	if (err)
		fprintf(stderr, "set max age failed: %s\n",
			strerror(err));

	return err != 0;
}

static int br_cmd_setpathcost(int argc, char *const* argv)
{
	int cost, err;

	if (sscanf(argv[3], "%i", &cost) != 1) {
		fprintf(stderr, "bad path cost value\n");
		return 1;
	}

	err = br_set_path_cost(argv[1], argv[2], cost);
	if (err)
		fprintf(stderr, "set path cost failed: %s\n",
			strerror(err));
	return err != 0;
}

static int br_cmd_setportprio(int argc, char *const* argv)
{
	int cost, err;

	if (sscanf(argv[3], "%i", &cost) != 1) {
		fprintf(stderr, "bad path priority value\n");
		return 1;
	}

	err = br_set_port_priority(argv[1], argv[2], cost);
	if (err)
		fprintf(stderr, "set port priority failed: %s\n",
			strerror(errno));

	return err != 0;
}

#ifdef ARCADYAN
#ifdef CONFIG_BRIDGE_UTILS_IGMP_SNP
static int br_cmd_enableportsnooping(int argc, char *const* argv)
{
	int err;
	int enable;

        sscanf(argv[2], "%i", &enable);

	if((enable < 0) || (enable > 2)) {
		fprintf(stderr, "bad value\n");
		return 1;
        }

	err = br_enable_port_snooping(argv[1], enable);
	if (err)
		fprintf(stderr, "enable port snooping failed: %s\n",
			strerror(errno));

	return err != 0;
}
#endif /* CONFIG_BRIDGE_UTILS_IGMP_SNP */

#ifdef CONFIG_BRIDGE_UTILS_IGMP_PROXY
static int br_cmd_enableproxymode(int argc, char *const* argv)
{
	int err;
	int enable;

        sscanf(argv[2], "%i", &enable);

	if((enable != 0) && (enable != 1)) {
		fprintf(stderr, "bad value\n");
		return 1;
	}

	err = br_enable_proxy_mode(argv[1], enable);
	if (err)
		fprintf(stderr, "enable proxy mode failed: %s\n",
			strerror(errno));

	return err != 0;
}
#endif /* CONFIG_BRIDGE_UTILS_IGMP_PROXY */

#ifdef CONFIG_BRIDGE_UTILS_MLD_SNP
static int br_cmd_mld_enableportsnooping(int argc, char*const* argv)
{
	int err;
	int enable;

        sscanf(argv[2], "%i", &enable);

	if((enable < 0) || (enable > 2)) {
		fprintf(stderr, "bad value\n");
		return 1;
	}

	err = br_mld_enable_port_snooping(argv[1], enable);
	if (err)
		fprintf(stderr, "enable mld port snooping failed: %s\n",
			strerror(errno));

	return err != 0;
}
#endif /* CONFIG_BRIDGE_UTILS_MLD_SNP */

#ifdef CONFIG_BRIDGE_UTILS_MLD_PROXY
static int br_cmd_mld_enableproxymode(int argc, char*const* argv)
{
	int err;
	int enable;

        sscanf(argv[2], "%i", &enable);

	if((enable != 0) && (enable != 1)) {
		fprintf(stderr, "bad value\n");
		return 1;
	}

	err = br_mld_enable_proxy_mode(argv[1], enable);
	if (err)
		fprintf(stderr, "enable mld proxy mode failed: %s\n",
			strerror(errno));

	return err != 0;
}
#endif /* CONFIG_BRIDGE_UTILS_MLD_PROXY */

#ifdef CONFIG_BRIDGE_UTILS_IGMP_RATE_LIMIT
static int br_cmd_enableigmpratelimit(int argc, char*const* argv)
{
	int err;
	int limit;

	sscanf(argv[2], "%i", &limit);

	if( limit > 500 )
	{
		fprintf(stderr, "bad value\n");
		return 1;
	}

	err = br_igmp_enable_rate_limit(argv[1], limit);
	if (err)
	{
		fprintf(stderr, "enable igmp rate limit failed: %s\n",
			strerror(errno));
	}

	return err != 0;
}

static int br_cmd_getportigmpcounter(int argc, char *const* argv)
{
	int err;
	unsigned long rate, recv, drop, total_drop, portId;


	sscanf(argv[1], "%lu", &portId);
	err = br_igmp_get_port_counters(portId, &rate, &recv, &drop, &total_drop);

	if (err)
		fprintf(stderr, "get igmp counter of port %d failed: %s\n", portId, strerror(errno));
	else {
		printf("Port %d\n", portId);
		printf("configured rate = %lu pps\n", rate);
		printf("received near a second = %lu\n", recv);
		printf("dropped near a second = %lu\n", drop);
		printf("total dropped = %lu\n", total_drop);
	}

	return err != 0;
}

static int br_cmd_setportigmpratelimit(int argc, char *const* argv)
{
	int err;
	unsigned long rate, portId;

	sscanf(argv[1], "%lu", &portId);
	sscanf(argv[2], "%lu", &rate);
	if (rate < 0 || portId < 0) {
		fprintf(stderr, "bad value\n");
		return 1;
	}

	err = br_igmp_set_port_rate_limit(portId, rate);
	if (err)
		fprintf(stderr, "set igmp rate limit of port %d failed: %s\n", portId, strerror(errno));

	return err != 0;
}
#endif /* CONFIG_BRIDGE_UTILS_IGMP_RATE_LIMIT */

#ifdef CONFIG_BRIDGE_UTILS_IGMP_FILTER
static int br_cmd_portigmpfilter(int argc, char *const* argv)
{
	int err, port, enable, action;

	if (sscanf(argv[1], "%d", &port) != 1) {
		fprintf(stderr, "port is 0..(uni#-1)\n");
		return 1;
	}

	if (argc == 2) {
		err = br_get_port_igmpfilter(port, &enable, &action);
		if (err) {
			fprintf(stderr, "get igmpfilter of port %d failed: %s\n", port, strerror(errno));
			return 1;
		}
		printf("Filter: %s\n", enable ? "enabled" : "disabled");
		printf("Default: %s\n", action ? "reject" : "accept");
		return 0;
	}

	if (sscanf(argv[2], "%d", &enable) != 1) {
		fprintf(stderr, "enable is 0(disable) or 1(enable)\n");
		return 1;
	}
	if (sscanf(argv[3], "%d", &action) != 1) {
		fprintf(stderr, "action is 0(accept) or 1(reject)\n");
		return 1;
	}


	err = br_set_port_igmpfilter(port, enable, action);
	if (err) {
		fprintf(stderr, "set igmpfilter of port %d failed: %s\n", port, strerror(errno));
	}

	return err != 0;
}

static int br_cmd_addportigmpfilterrule(int argc, char *const* argv)
{
	int err, port, index, startip[4], endip[4], action, i, shift;
	unsigned long start_ip, end_ip;

	if (sscanf(argv[1], "%d", &port) != 1) {
		fprintf(stderr, "port is 0..(uni#-1)\n");
		return 1;
	}
	if (sscanf(argv[2], "%d.%d.%d.%d", &startip[0], &startip[1], &startip[2], &startip[3]) != 4) {
		fprintf(stderr, "startip is xxx.xxx.xxx.xxx\n");
		return 1;
	}
	if (sscanf(argv[3], "%d.%d.%d.%d", &endip[0], &endip[1], &endip[2], &endip[3]) != 4) {
		fprintf(stderr, "endip is xxx.xxx.xxx.xxx\n");
		return 1;
	}
	start_ip = 0;
	end_ip = 0;
	for (i = 0; i < 4; i++) {
		shift = 8 * (3 - i);
		startip[i] &= 0xff;
		start_ip |= (startip[i] << shift);
		endip[i] &= 0xff;
		end_ip |= (endip[i] << shift);
	}
	if (start_ip > end_ip) {
		fprintf(stderr, "startip must be smaller than or equal to endip\n");
		return 1;
	}

	if (sscanf(argv[4], "%d", &action) != 1) {
		fprintf(stderr, "action is 0(accept) or 1(reject)\n");
		return 1;
	}

	err = br_add_port_igmpfilterrule(port, index, start_ip, end_ip, action);
	if (err) {
		fprintf(stderr, "add igmpfilter rule %s-%s of port %d default failed: %s\n", 
													argv[2], argv[3], port, strerror(errno));
	}

	return err != 0;
}

static int br_cmd_delportigmpfilterrule(int argc, char *const* argv)
{
	int err, port, index, startip[4], endip[4], action, i, shift;
	unsigned long start_ip, end_ip;

	if (sscanf(argv[1], "%d", &port) != 1) {
		fprintf(stderr, "port is 0..(uni#-1)\n");
		return 1;
	}
	if (sscanf(argv[2], "%d.%d.%d.%d", &startip[0], &startip[1], &startip[2], &startip[3]) != 4) {
		fprintf(stderr, "startip is xxx.xxx.xxx.xxx\n");
		return 1;
	}
	if (sscanf(argv[3], "%d.%d.%d.%d", &endip[0], &endip[1], &endip[2], &endip[3]) != 4) {
		fprintf(stderr, "endip is xxx.xxx.xxx.xxx\n");
		return 1;
	}
	start_ip = 0;
	end_ip = 0;
	for (i = 0; i < 4; i++) {
		shift = 8 * (3 - i);
		startip[i] &= 0xff;
		start_ip |= (startip[i] << shift);
		endip[i] &= 0xff;
		end_ip |= (endip[i] << shift);
	}
	if (start_ip > end_ip) {
		fprintf(stderr, "startip must be smaller than or equal to endip\n");
		return 1;
	}

	err = br_del_port_igmpfilterrule(port, index, start_ip, end_ip, action);
	if (err) {
		fprintf(stderr, "del igmpfilter rule %s-%s of port %d default failed: %s\n", 
													argv[2], argv[3], port, strerror(errno));
	}

	return err != 0;
}

static int br_cmd_showportigmpfilterrule(int argc, char *const* argv)
{
	int err, port, action, start_ip, end_ip;
	char startStr[INET_ADDRSTRLEN], endStr[INET_ADDRSTRLEN];
	struct in_addr addr;
	int i;

	if (sscanf(argv[1], "%d", &port) != 1) {
		fprintf(stderr, "port is 0..(uni#-1)\n");
		return 1;
	}

	printf("Port %d:\n", port);
	printf("Start           End             Action\n");
	printf("======================================\n");
	for (i = 0; i < BR_IGMP_PORT_FILTER_RULE_MAX; i++) {
		err = br_get_port_igmpfilterrule(port, i, &start_ip, &end_ip, &action);
		if (err)
			break;


		addr.s_addr = start_ip;
		inet_ntop(AF_INET, &addr, startStr);
		addr.s_addr = end_ip;
		inet_ntop(AF_INET, &addr, endStr);
		printf("%-15s %-15s %s\n", startStr, endStr, action ? "reject" : "accept");
	}

	return 0;
}
#endif /* CONFIG_BRIDGE_UTILS_IGMP_FILTER */

#ifdef CONFIG_BRIDGE_UTILS_IGMP_GROUP_LIMIT
static int br_cmd_portigmpmaxgroup(int argc, char *const* argv)
{
	int err;
	unsigned short max, curr, portId;
	int i;

	sscanf(argv[1], "%u", &portId);
	if (argc == 2) {
		err = br_get_port_igmp_max_group(portId, &max, &curr);
		if (err) {
			fprintf(stderr, "get igmp max groups of port %d failed: %s\n", portId, strerror(errno));

			return 1;
		}	
		printf("max groups = %u\n", max);
		printf("received groups = %u\n", curr);
	    return 0;
	}

	sscanf(argv[2], "%u", &max);
	if (max < 0 || portId < 0) {
		fprintf(stderr, "bad value\n");
		return 1;
	}

	err = br_set_port_igmp_max_group(portId, max);
	if (err)
		fprintf(stderr, "set igmp max groups of port %d failed: %s\n", portId, strerror(errno));

	return err != 0;
}
#endif /* CONFIG_BRIDGE_UTILS_IGMP_GROUP_LIMIT */
#endif /* ARCADYAN */

static int br_cmd_stp(int argc, char *const* argv)
{
	int stp, err;

	if (!strcmp(argv[2], "on") || !strcmp(argv[2], "yes") 
	    || !strcmp(argv[2], "1"))
		stp = 1;
	else if (!strcmp(argv[2], "off") || !strcmp(argv[2], "no") 
		 || !strcmp(argv[2], "0"))
		stp = 0;
	else {
		fprintf(stderr, "expect on/off for argument\n");
		return 1;
	}

	err = br_set_stp_state(argv[1], stp);
	if (err)
		fprintf(stderr, "set stp status failed: %s\n", 
			strerror(errno));
	return err != 0;
}

static int br_cmd_showstp(int argc, char *const* argv)
{
	struct bridge_info info;

	if (br_get_bridge_info(argv[1], &info)) {
		fprintf(stderr, "%s: can't get info %s\n", argv[1],
			strerror(errno));
		return 1;
	}

	br_dump_info(argv[1], &info);
	return 0;
}

static int show_bridge(const char *name, void *arg)
{
	struct bridge_info info;

	printf("%s\t\t", name);
	fflush(stdout);

	if (br_get_bridge_info(name, &info)) {
		fprintf(stderr, "can't get info %s\n",
				strerror(errno));
		return 1;
	}

	br_dump_bridge_id((unsigned char *)&info.bridge_id);
	printf("\t%s\t\t", info.stp_enabled?"yes":"no");

	br_dump_interface_list(name);
	return 0;
}

static int br_cmd_show(int argc, char *const* argv)
{
	printf("bridge name\tbridge id\t\tSTP enabled\tinterfaces\n");
	br_foreach_bridge(show_bridge, NULL);
	return 0;
}

static int compare_fdbs(const void *_f0, const void *_f1)
{
	const struct fdb_entry *f0 = _f0;
	const struct fdb_entry *f1 = _f1;

	return memcmp(f0->mac_addr, f1->mac_addr, 6);
}

static int br_cmd_showmacs(int argc, char *const* argv)
{
	const char *brname = argv[1];
#define CHUNK 128
	int i, n;
	struct fdb_entry *fdb = NULL;
	int offset = 0;

	for(;;) {
		fdb = realloc(fdb, (offset + CHUNK) * sizeof(struct fdb_entry));
		if (!fdb) {
			fprintf(stderr, "Out of memory\n");
			return 1;
		}
			
		n = br_read_fdb(brname, fdb+offset, offset, CHUNK);
		if (n == 0)
			break;

		if (n < 0) {
			fprintf(stderr, "read of forward table failed: %s\n",
				strerror(errno));
			return 1;
		}

		offset += n;
	}

	qsort(fdb, offset, sizeof(struct fdb_entry), compare_fdbs);

	printf("port no\tmac addr\t\tis local?\tageing timer\n");
	for (i = 0; i < offset; i++) {
		const struct fdb_entry *f = fdb + i;
		printf("%3i\t", f->port_no);
		printf("%.2x:%.2x:%.2x:%.2x:%.2x:%.2x\t",
		       f->mac_addr[0], f->mac_addr[1], f->mac_addr[2],
		       f->mac_addr[3], f->mac_addr[4], f->mac_addr[5]);
		printf("%s\t\t", f->is_local?"yes":"no");
		br_show_timer(&f->ageing_timer_value);
		printf("\n");
	}
	return 0;
}

#ifdef ARCADYAN
#ifdef CONFIG_BRIDGE_UTILS_ADDDEL_MAC
static int br_cmd_addmacs(int argc, char *const* argv)
{
	const char *brName;
	const char *ifName;
	int         err;

	brName = *++argv;
	ifName = *++argv;
	argc  -= 3; /* skip past the command name and the first 2 arguments */

	while (argc-- > 0)
	{
		const char *pMac = *++argv;
		err = br_add_fdb(brName, ifName, pMac);

		switch(err) {
		case 0:
			continue;

		case ENODEV:
			if (if_nametoindex(ifName) == 0)
				fprintf(stderr, "interface %s does not exist!\n", ifName);
			else
				fprintf(stderr, "bridge %s does not exist!\n", brName);
			break;

		default:
			fprintf(stderr, "can't add mac %s to bridge %s, port %s: %s\n",
				pMac, brName, ifName, strerror(err));
		}
		return 1;
	}

	return 0;
}

static int br_cmd_delmacs(int argc, char *const* argv)
{
	const char *brName;
	const char *ifName;
	int         err;

	brName = *++argv;
	ifName = *++argv;
	argc  -= 3; /* skip past the command name and the first 2 arguments */

	while (argc-- > 0)
	{
		const char *pMac = *++argv;
		err = br_del_fdb(brName, ifName, pMac);

		switch(err) {
		case 0:
			continue;

		case ENODEV:
			if (if_nametoindex(ifName) == 0)
				fprintf(stderr, "interface %s does not exist!\n", ifName);
			else
				fprintf(stderr, "bridge %s does not exist!\n", brName);
			break;

		default:
			fprintf(stderr, "can't add mac %s to bridge %s, port %s: %s\n",
				pMac, brName, ifName, strerror(err));
		}
		return 1;
	}

	return 0;
}
#endif /* CONFIG_BRIDGE_UTILS_ADDDEL_MAC */
#endif /* ARCADYAN */

static const struct command commands[] = {
	{ 1, "addbr", br_cmd_addbr, "<bridge>\t\tadd bridge" },
	{ 1, "delbr", br_cmd_delbr, "<bridge>\t\tdelete bridge" },
	{ 2, "addif", br_cmd_addif, 
	  "<bridge> <device>\tadd interface to bridge" },
	{ 2, "delif", br_cmd_delif,
	  "<bridge> <device>\tdelete interface from bridge" },
	{ 2, "setageing", br_cmd_setageing,
	  "<bridge> <time>\t\tset ageing time" },
	{ 2, "setbridgeprio", br_cmd_setbridgeprio,
	  "<bridge> <prio>\t\tset bridge priority" },
	{ 2, "setfd", br_cmd_setfd,
	  "<bridge> <time>\t\tset bridge forward delay" },
	{ 2, "sethello", br_cmd_sethello,
	  "<bridge> <time>\t\tset hello time" },
	{ 2, "setmaxage", br_cmd_setmaxage,
	  "<bridge> <time>\t\tset max message age" },
	{ 3, "setpathcost", br_cmd_setpathcost, 
	  "<bridge> <port> <cost>\tset path cost" },
	{ 3, "setportprio", br_cmd_setportprio,
	  "<bridge> <port> <prio>\tset port priority" },
#ifdef ARCADYAN
#ifdef CONFIG_BRIDGE_UTILS_IGMP_SNP
	{ 2, "enableportsnooping", br_cmd_enableportsnooping,
	  "<bridge> <value>\t0-disable 1-standard 2-blocking" },
#endif /* CONFIG_BRIDGE_UTILS_IGMP_SNP */
#ifdef CONFIG_BRIDGE_UTILS_IGMP_PROXY
	{ 2, "enableproxymode", br_cmd_enableproxymode,
	  "<bridge> <value> \tTo enable 1 or disable 0" },
#endif /* CONFIG_BRIDGE_UTILS_IGMP_PROXY */
#ifdef CONFIG_BRIDGE_UTILS_MLD_SNP
	{ 2, "mldenableportsnooping", br_cmd_mld_enableportsnooping,
	  "<bridge> <value>\t0-disable 1-standard 2-blocking" },
#endif /* CONFIG_BRIDGE_UTILS_MLD_SNP */
#ifdef CONFIG_BRIDGE_UTILS_MLD_PROXY
	{ 2, "mldenableproxymode", br_cmd_mld_enableproxymode,
	  "<bridge> <value>\tTo enable 1 or disable 0" },
#endif /* CONFIG_BRIDGE_UTILS_MLD_PROXY */
#ifdef CONFIG_BRIDGE_UTILS_IGMP_RATE_LIMIT
	{ 2, "enableigmpratelimit", br_cmd_enableigmpratelimit,
	  "<bridge> <value>\t0-disable, 1..500-packet rate" },
	{ 2, "setportigmpratelimit", br_cmd_setportigmpratelimit, "<port> <value>\t0-disable, 1..500-packet rate" },
	{ 1, "getportigmpcounters", br_cmd_getportigmpcounter, "<port>" },
#endif /* CONFIG_BRIDGE_UTILS_IGMP_RATE_LIMIT */
#ifdef CONFIG_BRIDGE_UTILS_IGMP_FILTER
	{ 1, "portigmpfilter", br_cmd_portigmpfilter,
	  "<port> <enable> [<default-action>]" },
	{ 4, "addportigmpfilterrule", br_cmd_addportigmpfilterrule,
	  "<port> <startip> <endip> <action>\t0-accept 1-reject" },
	{ 3, "addportigmpfilterrule", br_cmd_delportigmpfilterrule,
	  "<port> <startip> <endip>" },
	{ 1, "showportigmpfilterrule", br_cmd_showportigmpfilterrule,
	  "<port>" },
#endif /* CONFIG_BRIDGE_UTILS_IGMP_FILTER */
#ifdef CONFIG_BRIDGE_UTILS_IGMP_GROUP_LIMIT
	{ 0, "portigmpgroup", br_cmd_portigmpmaxgroup, "<port> [<value>]\t0-disable, 1...#-igmp group" },
#endif /* CONFIG_BRIDGE_UTILS_IGMP_GROUP_LIMIT */
#endif /* ARCADYAN */
	{ 0, "show", br_cmd_show, "\t\t\tshow a list of bridges" },
	{ 1, "showmacs", br_cmd_showmacs, 
	  "<bridge>\t\tshow a list of mac addrs"},
#ifdef CONFIG_BRIDGE_UTILS_ADDDEL_MAC
	{ 3, "addmacs", br_cmd_addmacs, 
	  "<bridge> <port> <mac>\t\tadd mac addresses to the bridge table"},
	{ 3, "delmacs", br_cmd_delmacs, 
	  "<bridge> <port> <mac>\t\tremove mac addresses from the bridge table"},
#endif /* CONFIG_BRIDGE_UTILS_ADDDEL_MACS */
	{ 1, "showstp", br_cmd_showstp, 
	  "<bridge>\t\tshow bridge stp info"},
	{ 2, "stp", br_cmd_stp,
	  "<bridge> {on|off}\tturn stp on/off" },
};

const struct command *command_lookup(const char *cmd)
{
	int i;

	for (i = 0; i < sizeof(commands)/sizeof(commands[0]); i++) {
		if (!strcmp(cmd, commands[i].name))
			return &commands[i];
	}

	return NULL;
}

void command_helpall(void)
{
	int i;

	for (i = 0; i < sizeof(commands)/sizeof(commands[0]); i++) {
		printf("\t%-10s\t%s\n", commands[i].name, commands[i].help);
	}
}
