/*
 * (C) Copyright 2002
 * Gerald Van Baren, Custom IDEAS, vanbaren@cideas.com
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

/*
 * SPI Read/Write Utilities
 */

#include <common.h>
#include <command.h>
#include <spi.h>

#if (CONFIG_COMMANDS & CFG_CMD_SPI)
#if !(defined(CONFIG_COMCERTO_100) || defined(CONFIG_COMCERTO_1000))

/*-----------------------------------------------------------------------
 * Definitions
 */

#ifndef MAX_SPI_BYTES
#   define MAX_SPI_BYTES 32	/* Maximum number of bytes we can handle */
#endif

/*
 * External table of chip select functions (see the appropriate board
 * support for the actual definition of the table).
 */
extern spi_chipsel_type spi_chipsel[];
extern int spi_chipsel_cnt;

/*
 * Values from last command.
 */
static int   device;
static int   bitlen;
static uchar dout[MAX_SPI_BYTES];
static uchar din[MAX_SPI_BYTES];

/*
 * SPI read/write
 *
 * Syntax:
 *   spi {dev} {num_bits} {dout}
 *     {dev} is the device number for controlling chip select (see TBD)
 *     {num_bits} is the number of bits to send & receive (base 10)
 *     {dout} is a hexadecimal string of data to send
 * The command prints out the hexadecimal string received via SPI.
 */

int do_spi (cmd_tbl_t *cmdtp, int flag, int argc, char *argv[])
{
	char  *cp = 0;
	uchar tmp;
	int   j;
	int   rcode = 0;

	/*
	 * We use the last specified parameters, unless new ones are
	 * entered.
	 */

	if ((flag & CMD_FLAG_REPEAT) == 0)
	{
		if (argc >= 2)
			device = simple_strtoul(argv[1], NULL, 10);
		if (argc >= 3)
			bitlen = simple_strtoul(argv[2], NULL, 10);
		if (argc >= 4) {
			cp = argv[3];
			for(j = 0; *cp; j++, cp++) {
				tmp = *cp - '0';
				if(tmp > 9)
					tmp -= ('A' - '0') - 10;
				if(tmp > 15)
					tmp -= ('a' - 'A');
				if(tmp > 15) {
					printf("Hex conversion error on %c, giving up.\n", *cp);
					return 1;
				}
				if((j % 2) == 0)
					dout[j / 2] = (tmp << 4);
				else
					dout[j / 2] |= tmp;
			}
		}
	}

	if ((device < 0) || (device >=  spi_chipsel_cnt)) {
		printf("Invalid device %d, giving up.\n", device);
		return 1;
	}
	if ((bitlen < 0) || (bitlen >  (MAX_SPI_BYTES * 8))) {
		printf("Invalid bitlen %d, giving up.\n", bitlen);
		return 1;
	}

	debug ("spi_chipsel[%d] = %08X\n",
		device, (uint)spi_chipsel[device]);

	if(spi_xfer(spi_chipsel[device], bitlen, dout, din) != 0) {
		printf("Error with the SPI transaction.\n");
		rcode = 1;
	} else {
		cp = (char *)din;
		for(j = 0; j < ((bitlen + 7) / 8); j++) {
			printf("%02X", *cp++);
		}
		printf("\n");
	}

	return rcode;
}

/***************************************************/

U_BOOT_CMD(
	sspi,	5,	1,	do_spi,
	"sspi    - SPI utility commands\n",
	"<device> <bit_len> <dout> - Send <bit_len> bits from <dout> out the SPI\n"
	"<device>  - Identifies the chip select of the device\n"
	"<bit_len> - Number of bits to send (base 10)\n"
	"<dout>    - Hexadecimal string that gets sent\n"
);

#endif  /* !defined(CONFIG_COMCERTO_100) */


/* =================================================================================================== */

#if defined(CONFIG_COMCERTO_100) || defined(CONFIG_COMCERTO_1000)

#define SPI_BASEADDR		0x10098000
#define SPI_CTRLR0_REG		(SPI_BASEADDR+0x00)
#define SPI_CTRLR1_REG		(SPI_BASEADDR+0x04)
#define SPI_SSIENR_REG		(SPI_BASEADDR+0x08)
#define SPI_MWCR_REG		(SPI_BASEADDR+0x0c)
#define SPI_SER_REG		(SPI_BASEADDR+0x10)
#define SPI_BAUDR_REG		(SPI_BASEADDR+0x14)
#define SPI_TXFTLR_REG		(SPI_BASEADDR+0x18)
#define SPI_RXFTLR_REG		(SPI_BASEADDR+0x1c)
#define SPI_TXFLR_REG		(SPI_BASEADDR+0x20)
#define SPI_RXFLR_REG		(SPI_BASEADDR+0x24)
#define SPI_SR_REG		(SPI_BASEADDR+0x28)
#define SPI_IMR_REG		(SPI_BASEADDR+0x2c)
#define SPI_ISR_REG		(SPI_BASEADDR+0x30)
#define SPI_RISR_REG		(SPI_BASEADDR+0x34)
#define SPI_TXOICR_REG		(SPI_BASEADDR+0x38)
#define SPI_RXOICR_REG		(SPI_BASEADDR+0x3c)
#define SPI_RXUICR_REG		(SPI_BASEADDR+0x40)
#define SPI_MSTICR_REG		(SPI_BASEADDR+0x44)
#define SPI_ICR_REG		(SPI_BASEADDR+0x48)
#define SPI_IDR_REG		(SPI_BASEADDR+0x58)
#define SPI_DR_REG		(SPI_BASEADDR+0x60)

#define MAX_SPI_BYTES 32	/* Maximum number of bytes we can handle */
#define BAUDRATE ((((165 * 1000000) / 12000000) + 1) & 0xFFFE)

#define SPI_READ_REG(reg)	*(volatile u32*)(reg)
#define SPI_WRITE_REG(reg, val)	*(volatile u32*)(reg) = val

static unsigned int baudrate = BAUDRATE;
static unsigned int ctrl0 = 0xC7;

/*
 * Values from last command.
 */
static int device = -1;
static int wcount;
static uchar wdata[MAX_SPI_BYTES];
static uchar rdata[MAX_SPI_BYTES];

static uchar do_spi_write_read(uchar wbyte)
{
	uchar rbyte;
	SPI_WRITE_REG(SPI_SSIENR_REG, 1);
	SPI_WRITE_REG(SPI_DR_REG, wbyte);
	while (SPI_READ_REG(SPI_RXFLR_REG) == 0)   /* wait for rx fifo not empty */
		;
	rbyte = SPI_READ_REG(SPI_DR_REG) & 0xFF;
	while (SPI_READ_REG(SPI_SR_REG) & 0x01)  /* wait for not busy */
		;
	SPI_WRITE_REG(SPI_SSIENR_REG, 0);

	return rbyte;
}


/*
 * SPI read/write
 *
 * Syntax:
 *   spi {dev} {wbyte0} {wbyte1} ...
 *     {dev} is the device number for controlling chip select
 *     {wbyteX} is write byte #X
 */

int do_spi (cmd_tbl_t *cmdtp, int flag, int argc, char *argv[])
{
	int   rcode = 0;
	int i;

	/*
	 * We use the last specified parameters, unless new ones are
	 * entered.
	 */

	if ((flag & CMD_FLAG_REPEAT) == 0)
	{
	    	device = -1;
		wcount = 0;
		if (argc >= 2)
			device = simple_strtoul(argv[1], NULL, 10);
		for (i = 2; i < argc; i++)
		{
			if (wcount == MAX_SPI_BYTES)
			{
				printf("Too many write bytes.\n");
				return 1;
			}
			wdata[wcount++] = simple_strtoul(argv[i], NULL, 16);
		}
	}

	if (wcount == 0)
	{
		printf("Invalid command arguments.\n");
		return 1;
	}

	while (SPI_READ_REG(SPI_SR_REG) & 0x01)  /* wait for not busy */
		;

	SPI_WRITE_REG(SPI_SSIENR_REG, 0);
	SPI_WRITE_REG(SPI_CTRLR0_REG, (0x00 << 8) | ctrl0);
	SPI_WRITE_REG(SPI_BAUDR_REG, baudrate);
	SPI_WRITE_REG(SPI_SER_REG, 1 << device);
	SPI_WRITE_REG(SPI_RXFTLR_REG, 8);
	SPI_WRITE_REG(SPI_TXFTLR_REG, 0);
	SPI_WRITE_REG(SPI_IMR_REG, 0);

	for (i = 0; i < wcount; i++)
	{
		rdata[i] = do_spi_write_read(wdata[i]);
	}
	for (i = 0; i < wcount; i++)
	{
		printf("%02X", rdata[i]);
		printf(i == wcount - 1 ? "\n" : " ");
	}

	return rcode;
}

/***************************************************/

U_BOOT_CMD(
	spi,	32,	1,	do_spi,
	"spi     - SPI utility commands\n",
	"<device> <wbyte0> <wbyte1> ... - Write/Read the SPI interface\n"
	"<device>  - Identifies the chip select of the device (0-3)\n"
	"<wbyteX>  - Write byte # X\n"
);

#endif  /* defined(CONFIG_COMCERTO_100) */

#endif	/* CFG_CMD_SPI */
