/*
 *  MIB group implementation - adslNodes.c
 */
#include <net-snmp/net-snmp-config.h>
#include <net-snmp/net-snmp-includes.h>

#include "common/cmmif.h" 
#include "common/writeMethod.h"

#include "adslNodes.h"

static unsigned long ulRecoverConfig = 0;
static unsigned long ulReboot = 0;
static unsigned long ulWhichAction = 0;

unsigned char *var_remoteSnmpEntry(struct variable *vp,
            					                  oid     *name,
            					                  size_t  *length,
            					                  int     exact,
            					                  size_t  *var_len,
            					                  WriteMethod **write_method);
unsigned char *var_localSntpEntry(struct variable *vp,
            					                  oid     *name,
            					                  size_t  *length,
            					                  int     exact,
            					                  size_t  *var_len,
            					                  WriteMethod **write_method);

struct variable1 remotesnmp_variables[] = {
    {OID_snmpEnabled, ASN_INTEGER, RWRITE, var_remoteSnmpEntry, 1, {1}}, 
    {OID_snmpReadCommunity, ASN_OCTET_STR, RWRITE, var_remoteSnmpEntry, 1, {2}}, 
    {OID_snmpWriteCommunity, ASN_OCTET_STR, RWRITE, var_remoteSnmpEntry, 1, {3}}, 
    {OID_snmpDescription, ASN_OCTET_STR, RONLY, var_remoteSnmpEntry, 1, {4}}, 
    {OID_snmpTrapServer, ASN_IPADDRESS, RWRITE, var_remoteSnmpEntry, 1, {5}}, 
    {OID_resetAP, ASN_INTEGER, RWRITE, var_remoteSnmpEntry, 1, {6}}, 
    {OID_restoreFactory, ASN_INTEGER, RWRITE, var_remoteSnmpEntry, 1, {7}}
};
oid remotesnmp_variables_oid[] = {IGD_OID_PREFIX, 7};

struct variable1 localsntp_variables[] = {
    {OID_timeserver, ASN_OCTET_STR, RWRITE, var_localSntpEntry, 1, {1}}, 
#if 0
    {OID_timeserverport, ASN_INTEGER, RWRITE, var_localSntpEntry, 1, {2}}, 
#endif
    {OID_currenttime, ASN_OCTET_STR, RONLY, var_localSntpEntry, 1, {3}}, 
    {OID_timezone, ASN_INTEGER, RWRITE, var_localSntpEntry, 1, {4}}, 
    {OID_daylightsaving, ASN_INTEGER, RWRITE, var_localSntpEntry, 1, {5}}
};
oid localsntp_variables_oid[] = {IGD_OID_PREFIX, 9};
char *cstTimeZoneMatched[] = {
"GMT-12:00", "0",
"GMT-11:00", "1",
"GMT-10:00", "2",
"GMT-09:00", "3",
"GMT-08:00", "4",
"GMT-07:00", "7",
"GMT-06:00", "9",
"GMT-05:00", "13",
"GMT-04:00", "15",
"GMT-03:00", "20",
"GMT-02:00", "22",
"GMT-01:00", "24",
"GMT-00:00", "26",
"GMT+01:00", "27",
"GMT+02:00", "37",
"GMT+03:00", "39",
"GMT+04:00", "43",
"GMT+05:00", "47",
"GMT+05:30", "48",
"GMT+06:00", "51",
"GMT+07:00", "54",
"GMT+08:00", "56",
"GMT+09:00", "61",
"GMT+10:00", "66",
"GMT+11:00", "71",
"GMT+12:00", "72"
};

static unsigned long p_ulAdslValue = 0;
static char p_szAdslValue[256] = {0};
static char p_szAdslPath[256] = {0};

void init_adslNodes(void)
{
    SNMPA_TRACE("Init adsl nodes ...\n");
    REGISTER_MIB("remotesnmp",  remotesnmp_variables, variable1, remotesnmp_variables_oid);
    REGISTER_MIB("localsntp",  localsntp_variables, variable1, localsntp_variables_oid);

    ulRecoverConfig = 0;
    ulReboot = 0;
    ulWhichAction = 0;
    Init_WriteMethodStruct();
}

void shutdown_adslNodes(void)
{
    Free_WriteMethodStruct();
}


int WriteMethod_Mgmt(int action,
                                          unsigned char *var_val,
                                          unsigned char var_val_type,
                                          size_t var_val_len, 
                                          unsigned char *statP, 
                                          oid *name, size_t length)
{
    switch(action)
    {
        case RESERVE1:
		{
	           unsigned long ulNewVal = *((unsigned long *) var_val);
	           if(var_val_len > sizeof(unsigned long)) return SNMP_ERR_WRONGLENGTH;
		    if(ulNewVal != 0 && ulNewVal != 1)return SNMP_ERR_WRONGVALUE;
        	}
		break;
        case RESERVE2:
		break;
	 case ACTION:
		if(ulWhichAction == 1)
		{
		    ulRecoverConfig = *((unsigned long *) var_val);
		    if(ulRecoverConfig == 1) CMM_RecoverConfig();
		}
		else if(ulWhichAction == 2)
		{
		    ulReboot = *((unsigned long *) var_val);
		    if(ulReboot == 1) CMM_Reboot();
		}
		else SNMPA_TRACE("Error occurs when handling mamange action to device!\n");
		break;
	 case UNDO:
		break;
        case COMMIT:
	 	break;
	 case FREE:
		break;
    }

    return NULL;
}
unsigned char *var_remoteSnmpEntry(struct variable *vp,
        					                  oid     *name,
        					                  size_t  *length,
        					                  int     exact,
        					                  size_t  *var_len,
        					                  WriteMethod **write_method)
{
    if(header_generic(vp, name, length, exact, var_len, write_method) ==  MATCH_FAILED)
        return NULL;

    switch(vp->magic)
    {
        case OID_snmpEnabled:
           *write_method = WriteMethod_Comm;
           Active_WriteMethod(vp->type, vp->acl, 1, 0, 1, NULL, TR069_ROOT_SNMP "Enable", NULL, 0);
	    if(CMM_GetUlong(TR069_ROOT_SNMP "Enable", &p_ulAdslValue, NULL, 0) == CMM_SUCCESS)
	    {
                return (unsigned char*)&p_ulAdslValue;
	    }
	    else return NULL;
        case OID_snmpReadCommunity:
           *write_method = WriteMethod_Comm;
           Active_WriteMethod(vp->type, vp->acl, 0, 32, 0, NULL, TR069_ROOT_SNMP "ROCommunity", NULL, 0);
	    if(CMM_GetStr(TR069_ROOT_SNMP "ROCommunity", p_szAdslValue, sizeof(p_szAdslValue), NULL, 0) == CMM_SUCCESS)
	    {
                *var_len = strlen(p_szAdslValue);
                return (unsigned char*)p_szAdslValue;
	    }
           return NULL;
        case OID_snmpWriteCommunity:
           *write_method = WriteMethod_Comm;
           Active_WriteMethod(vp->type, vp->acl, 0, 32, 0, NULL, TR069_ROOT_SNMP "RWCommunity", NULL, 0);
	    if(CMM_GetStr(TR069_ROOT_SNMP "RWCommunity", p_szAdslValue, sizeof(p_szAdslValue), NULL, 0) == CMM_SUCCESS)
	    {
                *var_len = strlen(p_szAdslValue);
                return (unsigned char*)p_szAdslValue;
	    }
           return NULL;
        case OID_snmpDescription:
	    if(CMM_GetUlong(TR069_ROOT_SNMP "TrapVersion", &p_ulAdslValue, NULL, 0) == CMM_SUCCESS)
	    {
                sprintf(p_szAdslValue, "Current Trap version is %lu.", p_ulAdslValue);
                *var_len = strlen(p_szAdslValue);
                return (unsigned char*)p_szAdslValue;
	    }
	    else return NULL;
        case OID_snmpTrapServer:
           *write_method = WriteMethod_Comm;
           Active_WriteMethod(vp->type, vp->acl, 0, 32, 0, NULL, TR069_ROOT_SNMP "TrapHost", NULL, 0);
	    if(CMM_GetStr(TR069_ROOT_SNMP "TrapHost", p_szAdslValue, sizeof(p_szAdslValue), NULL, 0) == CMM_SUCCESS)
	    {
               return CHARS_TO_intIPADDR(p_szAdslValue);
	    }
           return NULL;
        case OID_resetAP:
            *write_method = WriteMethod_Mgmt;
	     ulWhichAction = 2;
            return (unsigned char*)&ulReboot;
        case OID_restoreFactory:
            *write_method = WriteMethod_Mgmt;
	     ulWhichAction = 1;
            return (unsigned char*)&ulRecoverConfig;
        default:
            break;
    }
    
    return NULL;
}

static int SCA_getCurDate(char *szCurDate)
{
    int cnt = 0;
    FILE* fp = NULL;
    char cmd[16];
    char buff[48];

    strcpy(cmd, "date");
    fp = popen(cmd, "r");
    if (!fp)
    {
    	CMMIF_TRACE("err: open pipe of '%s' for read failed - %s!\n", cmd, strerror(errno));
    	return -1;
    }

    while (!feof(fp)) 
    {
    	if (!fgets(buff, sizeof(buff), fp))  /* ꣬ȻرļִепܷSIG_PIPE쳣 */
       {
    	    break;
    	}

       if(cnt ++ == 0) //we just return the first line
       {
            strncpy(szCurDate, buff, strlen(buff)-1); //-1 means the last char is LR
       }
    }
    pclose(fp);

    return strlen(szCurDate);
}
unsigned char *var_localSntpEntry(struct variable *vp,
            					                  oid     *name,
            					                  size_t  *length,
            					                  int     exact,
            					                  size_t  *var_len,
            					                  WriteMethod **write_method)
{
    if(header_generic(vp, name, length, exact, var_len, write_method) ==  MATCH_FAILED)
        return NULL;

    switch(vp->magic)
    {
        case OID_timeserver:
           *write_method = WriteMethod_Comm;
           Active_WriteMethod(vp->type, vp->acl, 0, 64, 0, NULL, TR069_ROOT_SNTP "NTPServer1", NULL, 0);
	    if(CMM_GetStr(TR069_ROOT_SNTP "NTPServer1", p_szAdslValue, sizeof(p_szAdslValue), NULL, 0) == CMM_SUCCESS)
	    {
                *var_len = strlen(p_szAdslValue);
                return (unsigned char*)p_szAdslValue;
	    }
           return NULL;
#if 0           
        case OID_timeserverport:
#endif
        case OID_currenttime:
            if(SCA_getCurDate(p_szAdslValue) > 0)
            {
                *var_len = strlen(p_szAdslValue);
                return (unsigned char*)p_szAdslValue;
            }
           return NULL;
        case OID_timezone:
           *write_method = WriteMethod_Comm;
           Active_WriteMethod(vp->type, vp->acl, 1, 0, 72, NULL, TR069_ROOT_SNTP "LocalTimeZoneName", cstTimeZoneMatched, GET_ARRAY_BOUNADRY(cstTimeZoneMatched));
	    if(CMM_GetUlong(TR069_ROOT_SNTP "LocalTimeZoneName", &p_ulAdslValue, 
                                        cstTimeZoneMatched, GET_ARRAY_BOUNADRY(cstTimeZoneMatched)) == CMM_SUCCESS)
	    {
                return (unsigned char*)&p_ulAdslValue;
	    }
	    else return NULL;
        case OID_daylightsaving:
           *write_method = WriteMethod_Comm;
           Active_WriteMethod(vp->type, vp->acl, 1, 0, 1, NULL, TR069_ROOT_SNTP "DaylightSavingsUsed", NULL, 0);
	    if(CMM_GetUlong(TR069_ROOT_SNTP "DaylightSavingsUsed", &p_ulAdslValue, NULL, 0) == CMM_SUCCESS)
	    {
                return (unsigned char*)&p_ulAdslValue;
	    }
	    else return NULL;
        default:
            break;
    }

    return NULL;
}
